; LIBF32.ASM

;		Bibliotheque mathematique pour les flottants sur 32 bits
;		--------------------------------------------------------
;
;						(C) E+E CHAMOUARD 2002

; La bibliotheque offre les sous-programmes mathematiques pour les nombres
;	flottants codes sur 32 bits (simple precision) selon le standard IEEE754.

; Pour les fonctions arithmetiques (addition, soustraction, multiplication et
;	division), les nombres en entree ne sont pas completement testes (flottant 
;	anormal, infini,...). C'est au programme utilisateur de faire le necessaire 
;	(lors du chargement en memoire par exemple) avec les fonctions mises a 
;	disposition (F32ValidREx, F32InfREx, F32PresREx et F32ZeroREx). En sortie 
;	la carry (C) indique au programme utilisateur si tout c'est bien passe 
;	(C=1 si probleme). Les fonctions mathematiques sont compatibles (et ont ete
;	testees sur les nombre anormaux). Consulter chaque fonction pour les details 
;	specifiques.

; Cette bibliotheque necessite l'utilisation de la bibliotheque sur entiers
;	32 bits. Par compatibilite avec les autres bibliotheques des fonctions
;	(ex. ecriture/lecture en memoire) sont aussi redefinies dans cette
;	bibliotheque.

; L'erreur de calcul ne depasse pas un LSB avec une mantisse de 24 bits soit
;	une precision meilleure que 10-7 (sauf nombres "anormaux").

; Rappel du standard IEEE754 (simple precision) :
; Les nombres (normaux) sont sous la forme s*(1.f)*2^x avec :
;	  s : le bit de signe (1=negatif)
;	  f : la mantisse sur 23 bits (24 bits de precision avec le 1 implicite)
;	  x : l'exposant sur 8 bits avec un offset de 127 (129 <=> 2^2)
; Les nombres anormaux sont sous la forme s*(0.f)*2^(x+1) (0 implicite)

; Exemples et codages speciaux :
;
;	s	 x		   f      hexas (i+3 a i)   Signification
;
;	0	127			  0		3f 80 00 00		+1
;	0	125		4194304		3e c0 00 00		+0.375
;	0	128			  0		40 00 00 00		+2
;	1	128			  0		c0 00 00 00		-2
;	0	127		1906731		3f 9d 18 2b		+1.2273
;	0	  0			  0		00 00 00 00		+0					Plus zero
;	1	  0			  0		80 00 00 00		-0					Moins zero
;	0	255			  0		7f 80 00 00		+INF				Plus l'infini
;	1	255			  0		ff 80 00 00		-INF				Moins l'infini
;	0	254		8388607		7f 7f ff ff		+3.4028233e+38		Plus grand positif
;	1	254		8388607		ff 7f ff ff		-3.4028233e+38		Plus grand negatif
;	0	  1			  0		00 80 00 00		+1.1754944e-38		Plus petit positif normal
;	1	  1			  0		00 80 00 80		-1.1754944e-38		Plus petit negatif normal
;	0	255			!=0		7f xx xx xx		+NaN (Not a Number)
;	1	255			!=0		ff xx xx xx		-NaN (Not a Number)
;	0	  0			!=0		00 xx xx xx		+Denormal number (gradual undeflow)
;	1	  0			!=0		80 xx xx xx		-Denormal number (gradual undeflow)
;	0	  0			  1		00 00 00 01		+1.4012985e-45		Plus petit positif anormal
;	0	  0			256		00 00 01 00		+3.5873241e-43
;	0	  0		  65536		00 01 00 00		+9.1835496e-41
;	0	  0		8388607		00 7f ff ff		+1.1754942e-38		Plus grand positif anormal


; La bibliotheque travaille sur 3 groupes de registres :
;	RE1x pour le premier operande ou premier resultat,
;	RE2x pour le second operande ou second resultat,
;	TMPx pour les resultats intermediaires.

; Le programme principal doit donc definir 12 registres directs :
;	direct  RE10, xxh   ; LSB du premier operande ou premier resultat
;	direct  RE11, xxh   ;
;	direct  RE12, xxh   ;
;	direct  RE13, xxh   ; MSB du premier operande ou premier resultat
;	direct  RE20, xxh   ; LSB du second operande ou second resultat
;	direct  RE21, xxh   ;
;	direct  RE22, xxh   ;
;	direct  RE23, xxh   ; MSB du second operande ou second resultat
;	direct  TMP0, xxh   ; LSB du resultat intermediaire
;	direct  TMP1, xxh   ;
;	direct  TMP2, xxh   ;
;	direct  TMP3, xxh   ; MSB du resultat intermediaire

; Le chargement ou sauvegarde des resultats ou des operandes s'effectue
;	par chargement  :
;		dans la memoire externe   (ex : F32ReadExtREx),
;		dans la memoire programme (ex : F32ReadProgREx),
;		dans la memoire interne   (ex : F32ReadIntREx).

; La carry (C) est positionnee en cas de debordement ou d'erreur (division)
;	pour tous les sous-programmes sauf ceux de chargement ou de sauvegarde.

; Liste des sous-programmes :
;	F32ReadExtRE1				Lecture de RE1 depuis la memoire externe
;	F32ReadExtRE2				Lecture de RE2 depuis la memoire externe
;	F32ReadProgRE1				Lecture de RE1 depuis la memoire programme
;	F32ReadProgRE2				Lecture de RE2 depuis la memoire programme
;	F32ReadIntRE1				Lecture de RE1 depuis la memoire interne
;	F32ReadIntRE2				Lecture de RE2 depuis la memoire interne
;	F32WriteExtRE1				Ecriture de RE1 dans la memoire externe
;	F32WriteExtRE2				Ecriture de RE2 dans la memoire externe
;	F32WriteIntRE1				Ecriture de RE1 dans la memoire interne
;	F32WriteIntRE2				Ecriture de RE2 dans la memoire interne
;	F32ClearRE1					Mise a 0 de RE1
;	F32Add						RE1=RE1+RE2
;	F32Sub						RE1=RE1-RE2
;	F32Mul						RE1=RE1*RE2
;	F32Div						RE1=RE1/RE2
;	F32NegRE1					RE1=-RE1
;	F32SgnRE1					C=Signe de RE1
;	F32AbsRE1					RE1=Valeur absolue de RE1
;	F32ValidRE1					C=0 si RE1!=NaN (Not a Number)
;	F32ValidRE2					C=0 si RE2!=NaN (Not a Number)
;	F32InfRE1					C=0 si RE1!=INF (Infinity)
;	F32InfRE2					C=0 si RE2!=INF (Infinity)
;	F32PresRE1					C=0 si RE1!=nombre avec gradual underflow
;	F32PresRE2					C=0 si RE2!=nombre avec gradual underflow
;	F32ZeroRE1					C=0 si RE1!=0
;	F32ZeroRE2					C=0 si RE2!=0
;	F32Cvti32tof32				Conversion signee de RE1 entier -> flottant
;	F32Cvtf32toi32				Conversion signee de RE1 flottant -> entier
;	F32UCvti32tof32				Conversion non signee de RE1 entier -> flottant
;	F32UCvtf32toi32				Conversion non signee de RE1 flottant -> entier

label   F32ReadExtRE1, I32ReadExtRE1
;
; Titre     :   Lecture de 32 bits de la memoire externe vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ReadExtRE2, I32ReadExtRE2
;
; Titre     :   Lecture de 32 bits de la memoire externe vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ReadProgRE1, I32ReadProgRE1
;
; Titre     :   Lecture de 32 bits de la memoire programme vers l'operande/resultat RE1
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ReadProgRE2, I32ReadProgRE2
;
; Titre     :   Lecture de 32 bits de la memoire programme vers l'operande/resultat RE2
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ReadIntRE1, I32ReadIntRE1
;
; Titre     :   Lecture de 32 bits de la memoire interne vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@R0+3,@R0+2,@R0+1,@R0} (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ReadIntRE2, I32ReadIntRE2
;
; Titre     :   Lecture de 32 bits de la memoire interne vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@R0+3,@R0+2,@R0+1,@R0} (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32WriteExtRE1, I32WriteExtRE1
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE1 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+3,@DPTR+2,@DPTR+1,@DPTR}=RE1 (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32WriteExtRE2, I32WriteExtRE2
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE2 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+3,@DPTR+2,@DPTR+1,@DPTR}=RE2 (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32WriteIntRE1, I32WriteIntRE1
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE1 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+3,@R0+2,@R0+1,@R0}=RE1 (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32WriteIntRE2, I32WriteIntRE2
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE2 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+3,@R0+2,@R0+1,@R0}=RE2 (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

label   F32ClearRE1, I32ClearRE1
;
; Titre     :   RAZ de l'operande/resultat RE1
;               ------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   RE1={0,0,0,0}=+0 (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

F32Add:
;
; Titre     :   Addition de deux nombres flottants de 32 bits
;               ---------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 perdu)
;				Si RE2=0 sortie de RE1 avec C=0
;				Sinon
;				  Si RE1=0 sortie de RE2 avec C=0
;					Sinon
;					  Si RE1=INF ou NaN sortie de RE1 avec C=1
;					  Sinon
;						Si RE2=INF ou NaN sortie de RE2 avec C=1
;						Sinon
;						  Si pas debordement sortie de RE1+RE2 avec C=0
;						  Sinon sortie de + ou - INF avec C=1 
; Utilise   :   Registres TMP0 a TMP3, A, B, F0 et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   F32ZeroRE2              ; Test arguments nuls pour accelerer les operations
jnc     F32Add1
clr		C                       ; RE2=0 donc retour direct de RE1
ret
F32Add1:
lcall   F32ZeroRE1
jnc     F32Add2
clr     C                       ; RE1=0 donc retour de RE2
ljmp    F32RE22RE1

F32Add2:                        ; Addition de deux nombres non nuls
lcall   F32GetexpRE1
mov     TMP0,A                  ; TMP0=exposant de RE1
lcall   F32GetexpRE2
mov     TMP1,A                  ; TMP1=exposant de RE2
mov		TMP2,RE13               ; TMP2.7=signe de RE1
mov		TMP3,RE23				; TMP3.7=signe de RE2
mov		A,TMP0
cpl		A
jnz		F32Add3
setb	C						; Retour de RE1 car RE1=infini (ou NAN)
ret
F32Add3:
mov		A,TMP1
cpl		A
jnz		F32Add4
setb	C						; Retour de RE2 car RE2=infini (ou NAN)
ljmp	F32RE22RE1

F32Add4:
mov     A,TMP0                  ; Test exp1>exp2+25
clr     C
subb    A,TMP1
jc      F32Add5                 ; Si exp1<exp2 suite
subb    A,#26
jc      F32Add6
ret                             ; Retour de RE1 car exp1>exp2+25
F32Add5:
mov     A,TMP1                  ; Test exp2>exp1+25
clr     C
subb    A,TMP0
jc      F32Add6                 ; Si exp2<exp1 suite
subb    A,#26
jc      F32Add6
ljmp    F32RE22RE1              ; Retour de RE2 car exp2>exp1+25
F32Add6:
mov		RE13,#00h				; Extraction des mantisses
mov		A,TMP0
jz		F32Add7 				; Saute si gradual underflow
orl		RE12,#80h				; Remise du 1 implicite
dec		TMP0
F32Add7:
mov		RE23,#00h
mov		A,TMP1
jz		F32Add8 				; Saute si gradual underflow
orl		RE22,#80h				; Remise du 1 implicite
dec		TMP1

F32Add8:						; Ici TMP0 et TMP1 vont de 0 a fd (-1 % IEEE754)
mov		A,#6
lcall	I32ShLeRE1				; x 64 pour ameliorer la precision
mov		A,#6
lcall	I32ShLeRE2				; x 64 pour ameliorer la precision
mov     A,TMP0                  ; Decalage du plus petit nombre vers la doite
clr     C
subb    A,TMP1
jc      F32Add9
lcall   I32ShRiRE2              ; Decal de mant2 de exp1-exp2
ljmp    F32Add10
F32Add9:                        ; exp2>exp1
cpl     A
inc     A
lcall   I32ShRiRE1              ; Decal de mant1 de exp2-exp1
mov     TMP0,TMP1				; Exposant dans TMP0
F32Add10:                       ; Remise du signe sur la mantisse 1
mov     A,TMP2
jnb     ACC.7,F32Add11
lcall   I32NegRE1
F32Add11:                       ; Remise du signe sur la mantisse 2
mov     A,TMP3
jnb     ACC.7,F32Add12
lcall   I32NegRE2

F32Add12: 
lcall   I32Add                  ; Les nombres sont synchros donc addition
mov     A,RE13                  ; Sortie du signe de la mantisse
mov     C,ACC.7
mov		F0,C					; Le signe est dans F0
inc		TMP0					; TMP0 remis sous sa forme IEEE754
mov		TMP1,#0					; 0 dans TMP1
jnb     ACC.7,F32Add13
lcall   I32NegRE1
F32Add13:						; TMP1,TMP0= exposant IEEE754, mantisse*64
ljmp	F32NormRE1				; Normalisation et sortie

F32Sub:
;
; Titre     :   Soustraction de deux nombres flottants de 32 bits
;               -------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 perdu)
;				Si RE2=0 sortie de RE1 avec C=0
;				Sinon
;				  Si RE1=0 sortie de -RE2 avec C=0
;					Sinon
;					  Si RE1=INF ou NaN sortie de RE1 avec C=1
;					  Sinon
;						Si RE2=INF ou NaN sortie de -RE2 avec C=1
;						Sinon
;						  Si pas debordement sortie de RE1-RE2 avec C=0
;						  Sinon sortie de + ou - INF avec C=1 
; Utilise   :   Registres TMP0 a TMP3, A, B, F0 et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
cpl     ACC.7
mov     RE23,A
ljmp    F32Add

F32Mul:
;
; Titre     :   Multiplication de deux nombres flottants de 32 bits
;               ---------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2 perdu)
;				Si RE1=0 sortie de 0 avec C=0
;				Sinon
;				  Si RE2=0 sortie de 0 avec C=0
;					Sinon
;					  Si RE1=INF ou NaN sortie de RE1 avec C=1
;					  Sinon
;						Si RE2=INF ou NaN sortie de RE2 avec C=1
;						Sinon
;						  Si pas debordement sortie de RE1*RE2 avec C=0
;						  Sinon sortie de + ou - INF avec C=1 
; Utilise   :   Registres TMP0 a TMP3, A, B, F0 et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   F32ZeroRE1              ; Test arguments nuls pour accelerer les operations
jnc     F32Mul1
ljmp	F32ClearRE1				; RE1=0 donc retour de 0
F32Mul1:
lcall   F32ZeroRE2
jnc     F32Mul2
ljmp	F32ClearRE1             ; RE2=0 donc retour de 0

F32Mul2:
lcall   F32GetexpRE1            ; Multiplication de deux nombres non nuls
mov     TMP0,A                  ; TMP0=exposant de RE1
lcall   F32GetexpRE2
mov     TMP1,A                  ; TMP1=exposant de RE2
mov		A,RE13
xrl		A,RE23
mov		C,ACC.7					; C=signe du produit
mov		F0,C					; F0=signe du produit
mov		A,TMP0
cpl		A
jnz		F32Mul3
mov		A,RE13
mov		ACC.7,C
mov		RE13,A
setb	C						; Retour de + ou - RE1 car RE1=infini (ou NAN)
ret
F32Mul3:
mov		A,TMP1
cpl		A
jnz		F32Mul4
mov		A,RE23
mov		ACC.7,C
mov		RE23,A
setb	C						; Retour de + ou - RE2 car RE2=infini (ou NAN)
ljmp	F32RE22RE1

F32Mul4:
mov		RE13,#00h				; Extraction des mantisses
mov		A,TMP0
jz		F32Mul5 				; Saute si gradual underflow
orl		RE12,#80h				; Remise du 1 implicite
dec		TMP0
F32Mul5:
mov		RE23,#00h
mov		A,TMP1
jz		F32Mul6 				; Saute si gradual underflow
orl		RE22,#80h				; Remise du 1 implicite
dec		TMP1

F32Mul6:						; Ici TMP0 et TMP1 vont de 0 a fd
mov		TMP3,RE12
mov		TMP2,RE11
mov		RE23,RE10				; Sauvegarde de la mantisse de RE1 un peu partout

mov		A,TMP3
mov		B,RE22
mul		AB
mov		RE13,B
mov		RE12,A					; RE13,RE12=RE12*RE22

mov		A,TMP2
mov		B,RE21
mul		AB
mov		RE11,B
mov		RE10,A					; RE11,RE10=RE11*RE21

mov		A,TMP3
mov		B,RE21
mul		AB
add		A,RE11
mov		RE11,A
mov		A,B
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11+=RE12*RE21

mov		A,TMP2
mov		B,RE22
mul		AB
add		A,RE11
mov		RE11,A
mov		A,B
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11+=RE11*RE22

mov		A,TMP3
mov		B,RE20
mul		AB
add		A,RE10
mov		RE10,A
mov		A,B
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11,RE10+=RE12*RE20

mov		A,TMP2
mov		B,RE20
mul		AB
mov		TMP3,A					; TMP3=RE1(-1)
mov		A,B
add		A,RE10
mov		RE10,A
mov		A,#0
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11,RE10,RE1(-1)+=RE11*RE20

mov		A,RE23
mov		B,RE22
mul		AB
add		A,RE10
mov		RE10,A
mov		A,B
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11,RE10+=RE10*RE22

mov		A,RE23
mov		B,RE21
mul		AB
add		A,TMP3
mov		TMP3,A
mov		A,B
addc	A,RE10
mov		RE10,A
mov		A,#0
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11,RE10,RE1(-1)+=RE10*RE21

mov		A,RE23
mov		B,RE20
mul		AB
mov		TMP2,A					; TMP2=RE1(-2)
mov		A,B
addc	A,TMP3
mov		TMP3,A
mov		A,#0
addc	A,RE10
mov		RE10,A
mov		A,#0
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; RE13,RE12,RE11,RE10,RE1(-1),RE1(-2)+=RE10*RE20

mov		A,TMP0
add		A,TMP1
mov		TMP0,A
mov		A,#0
rlc		A
mov		TMP1,A					; TMP1,TMP0=exp1+exp2
mov		A,TMP0
subb	A,#126
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A					; TMP1,TMP0=exp1+exp2-126

mov		A,RE13					; Normalisation si besoin pour ne pas perdre sur nombre
jnz		F32Mul7					; Ne sert normalement a rien
mov		RE13,RE12
mov		RE12,RE11
mov		RE11,RE10
mov		RE10,TMP3
mov		TMP3,TMP2
clr		C
mov		A,TMP0
subb	A,#8
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A

mov		A,RE13					; Normalisation si besoin pour ne pas perdre sur nombre
jnz		F32Mul7					; Ne sert normalement a rien
mov		RE13,RE12
mov		RE12,RE11
mov		RE11,RE10
mov		RE10,TMP3
clr		C
mov		A,TMP0
subb	A,#8
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A

F32Mul7:						; Mantisse sur 32 bits
ljmp	F32NormRE1

F32Div:
;
; Titre     :   Division de deux nombres flottants de 32 bits
;               ---------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 (RE2 perdu)
;				Si RE2=0 sortie de + ou - INF avec C=1
;				Sinon
;				  Si RE2=INF ou NAN sortie de + ou - 0 avec C=1
;					Sinon
;					  Si RE1=+ ou - 0 sortie de + ou - 0 avec C=0
;					  Sinon
;						Si RE1=INF ou NaN sortie de + ou - INF avec C=1
;						Sinon
;						  Si pas debordement sortie de RE1/RE2 avec C=0
;						  Sinon sortie de + ou - INF avec C=1 
; Utilise   :   Registres TMP0 a TMP3, A, B, F0 et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   F32GetexpRE1
mov     TMP0,A                  ; TMP0=exposant de RE1
lcall   F32GetexpRE2
mov     TMP1,A                  ; TMP1=exposant de RE2
mov		A,RE13
xrl		A,RE23
mov		C,ACC.7
mov		F0,C					; F0 est donc le signe du resultat

mov		A,RE23
anl		A,#01111111b
orl		A,RE22
orl		A,RE21
orl		A,RE20
jnz		F32Div1
ljmp	F32OutInf				; Denominateur nul, sortie + ou - infini

F32Div1:						; Denominateur non nul
mov		A,TMP1
cjne	A,#ffh,F32Div2
clr		A						; Denominateur infini (ou NaN)
mov		RE10,A					; Sortie de + ou - 0 avec C=1
mov		RE11,A
mov		RE12,A
mov		C,F0
mov		ACC.7,C
mov		RE13,A
setb	C
ret
F32Div2:

mov		A,RE13
anl		A,#01111111b
orl		A,RE12
orl		A,RE11
orl		A,RE10
jnz		F32Div3
clr		A						; Numerateur nul
mov		RE10,A					; Sortie de + ou - 0 avec C=0
mov		RE11,A
mov		RE12,A
mov		C,F0
mov		ACC.7,C
mov		RE13,A
clr		C
ret
F32Div3:
mov		A,TMP0
cjne	A,#ffh,F32Div4
ljmp	F32OutInf				; Num inf (ou NaN)+dem !nul => + ou - inf avec C=1
F32Div4:
		
mov		RE13,#00h				; Extraction des mantisses
mov		A,TMP0
jz		F32Div5 				; Saute si gradual underflow
orl		RE12,#80h				; Remise du 1 implicite
dec		TMP0
F32Div5:
mov		RE23,#00h
mov		A,TMP1
jz		F32Div6 				; Saute si gradual underflow
orl		RE22,#80h				; Remise du 1 implicite
dec		TMP1

F32Div6:						; Ici TMP0 et TMP1 vont de 0 a fd
clr		C
mov		A,TMP0
subb	A,TMP1
mov		TMP0,A
mov		A,#0
subb	A,#0
mov		TMP1,A					; TMP1,TMP0=exp1-exp2
mov		A,TMP0
add		A,#127
mov		TMP0,A
mov		A,TMP1
addc	A,#0
mov		TMP1,A					; TMP1,TMP0=exp1-exp2+127

F32Div7:
mov		A,RE22
jb		ACC.7,F32Div8			; mant2 est bien sur 24 bits exactement
mov		A,#1					; mant2 est sur moins de 24 bits
lcall	I32ShLeRE2				; => mant2=2*mant2
clr		C
mov		A,TMP0
subb	A,#1
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A
sjmp	F32Div7
F32Div8:						; il faut mant1>=mant2 pour obtenir la precision
clr		C
mov		A,RE10
subb	A,RE20
mov		A,RE11
subb	A,RE21
mov		A,RE12
subb	A,RE22
mov		A,RE13
subb	A,RE23
jnc		F32Div9					; mant1 >= mant2
mov		A,#1					; mant1 < mant2
lcall	I32ShLeRE1				; mant1=2*mant1
clr		C
mov		A,TMP0
subb	A,#1
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A
sjmp	F32Div8
F32Div9:						; mant2 sur 24 bits, mant1 sur max 25 bits
mov		A,#5
lcall	I32ShLeRE1
mov		A,#5
lcall	I32ShLeRE2				; mantisse * 32 pour la precision

push	TMP0
push	TMP1
mov		TMP3,RE13
mov		TMP2,RE12
mov		TMP1,RE11
mov		TMP0,RE10				; mant1 dans (TMP3,TMP2,TMP1,TMP0)
mov		RE10,#0
mov		B,#30
F32Div10:
clr		C
mov		A,TMP0
subb	A,RE20
mov		A,TMP1
subb	A,RE21
mov		A,TMP2
subb	A,RE22
mov		A,TMP3
subb	A,RE23
jc		F32Div11				; mant1 < mant2 donc rien a faire
clr		C						; mant1 >= mant2 donc mant1-=mant2
mov		A,TMP0
subb	A,RE20
mov		TMP0,A
mov		A,TMP1
subb	A,RE21
mov		TMP1,A
mov		A,TMP2
subb	A,RE22
mov		TMP2,A
mov		A,TMP3
subb	A,RE23
mov		TMP3,A
F32Div11:
cpl		C						; calcul du nouveau resultat
mov		A,RE10
rlc		A
mov		RE10,A
mov		A,RE11
rlc		A
mov		RE11,A
mov		A,RE12
rlc		A
mov		RE12,A
mov		A,RE13
rlc		A
mov		RE13,A
clr		C						; mant1=2*mant1
mov		A,TMP0
rlc		A
mov		TMP0,A
mov		A,TMP1
rlc		A
mov		TMP1,A
mov		A,TMP2
rlc		A
mov		TMP2,A
mov		A,TMP3
rlc		A
mov		TMP3,A
djnz	B,F32Div10
pop		TMP1
pop		TMP0
ljmp	F32NormRE1

F32NegRE1:
;
; Titre     :   Changement de signe de l'operande/resultat RE1
;               ----------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=-RE1
;               C=0 (pas d'erreur possible)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
cpl     ACC.7
mov     RE13,A
clr     C
ret

label   F32SgnRE1,I32SgnRE1
;
; Titre     :   Recuperation du signe de l'operande/resultat RE1
;               ------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si le nombre est positif
;               C=1 si le nombre est negatif
;               RE1 reste inchange
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;

F32AbsRE1:
;
; Titre     :   Valeur absolue de l'operande/resultat RE1
;               -----------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=ABS(RE1)
;               C=0 (pas d'erreur possible)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
clr     ACC.7
mov     RE13,A
clr     C
ret

F32ValidRE1:
;
; Titre     :   Verification de la validite de l'operande/resultat RE1
;               ------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si RE1 ne contient pas NAN et uniquement ce cas,
;               sinon C=1 (RE1 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
anl     A,#7fh
cjne    A,#7fh,F32ValidRE1End
mov     A,RE12                  ; RE13=7fh ou ffh
jnb     ACC.7,F32ValidRE1End
anl     A,#7fh                  ; RE12=1xxxxxxxb donc NAN ou INF
jnz     F32ValidRE1End2
mov     A,RE11                  ; RE12=80h
jnz     F32ValidRE1End2
mov     A,RE10                  ; RE11=00h
jnz     F32ValidRE1End2         ; Sinon RE10=00h donc + ou - INF
F32ValidRE1End:
clr     C
ret
F32ValidRE1End2:
setb    C
ret

F32ValidRE2:
;
; Titre     :   Verification de la validite de l'operande/resultat RE2
;               ------------------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   C=0 si RE2 ne contient pas NAN et uniquement ce cas,
;               sinon C=1 (RE2 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
anl     A,#7fh
cjne    A,#7fh,F32ValidRE2End
mov     A,RE22                  ; RE23=7fh ou ffh
jnb     ACC.7,F32ValidRE2End
anl     A,#7fh                  ; RE22=1xxxxxxxb donc NAN ou INF
jnz     F32ValidRE2End2
mov     A,RE21                  ; RE22=80h
jnz     F32ValidRE2End2
mov     A,RE20                  ; RE21=00h
jnz     F32ValidRE2End2         ; Sinon RE20=00h donc + ou - INF
F32ValidRE2End:
clr     C
ret
F32ValidRE2End2:
setb    C
ret

F32InfRE1:
;
; Titre     :   Verification de la non infinite de l'operande/resultat RE1
;               ----------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si RE1 ne contient pas +INF ni -INF et uniquement ces cas,
;               sinon C=1 (RE1 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
anl     A,#7fh
cjne    A,#7fh,F32InfRE1End
mov     A,RE12                  ; RE13=7fh ou ffh
cjne    A,#80h,F32InfRE1End
mov     A,RE11                  ; RE12=80h
jnz     F32InfRE1End
mov     A,RE10                  ; RE11=00h
jnz     F32InfRE1End
setb    C                       ; RE10=00h donc + ou - INF
ret
F32InfRE1End:
clr     C
ret

F32InfRE2:
;
; Titre     :   Verification de la non infinite de l'operande/resultat RE2
;               ----------------------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   C=0 si RE2 ne contient pas +INF ni -INF et uniquement ces cas,
;               sinon C=1 (RE2 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
anl     A,#7fh
cjne    A,#7fh,F32InfRE2End
mov     A,RE22                  ; RE23=7fh ou ffh
cjne    A,#80h,F32InfRE2End
mov     A,RE21                  ; RE22=80h
jnz     F32InfRE2End
mov     A,RE20                  ; RE21=00h
jnz     F32InfRE2End
setb    C                       ; RE20=00h donc + ou - INF
ret
F32InfRE2End:
clr     C
ret

F32PresRE1:
;
; Titre     :   Verification de la precision de l'operande/resultat RE1
;               -------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si RE1 ne contient pas un flottant avec gradual underflow
;               et uniquement ce cas,
;               sinon C=1 (RE1 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
anl     A,#7fh
jnz     F32PresRE1End
mov     A,RE12                  ; RE13=00h ou 80h
jb      ACC.7,F32PresRE1End     ; Sinon underflow ou 0
jnz     F32PresRE1End2          ; RE12=0xxxxxxxb
mov     A,RE11                  ; RE12=00h
jnz     F32PresRE1End2
mov     A,RE10                  ; RE11=00h
jnz     F32PresRE1End2          ; Sinon RE10=00h donc +0 ou -0
F32PresRE1End:
clr     C
ret
F32PresRE1End2:
setb    C
ret

F32PresRE2:
;
; Titre     :   Verification de la precision de l'operande/resultat RE2
;               -------------------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   C=0 si RE2 ne contient pas un flottant avec gradual underflow
;               et uniquement ce cas,
;               sinon C=1 (RE2 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
anl     A,#7fh
jnz     F32PresRE2End
mov     A,RE22                  ; RE23=00h ou 80h
jb      ACC.7,F32PresRE2End     ; Sinon underflow ou 0
jnz     F32PresRE2End2          ; RE22=0xxxxxxxb
mov     A,RE21                  ; RE22=00h
jnz     F32PresRE2End2
mov     A,RE20                  ; RE21=00h
jnz     F32PresRE2End2          ; Sinon RE20=00h donc +0 ou -0
F32PresRE2End:
clr     C
ret
F32PresRE2End2:
setb    C
ret

F32ZeroRE1:
;
; Titre     :   Verification de la non nullite de l'operande/resultat RE1
;               ---------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si RE1 ne contient pas +0 ni -0 et uniquement ces cas,
;               sinon C=1 (RE1 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
anl     A,#7fh
jnz     F32ZeroRE1End
mov     A,RE12                  ; RE13=00h ou 80h
jnz     F32ZeroRE1End
mov     A,RE11                  ; RE12=00h
jnz     F32ZeroRE1End
mov     A,RE10                  ; RE11=00h
jnz     F32ZeroRE1End
setb    C                       ; RE10=00h donc +0 ou -0
ret
F32ZeroRE1End:
clr     C
ret

F32ZeroRE2:
;
; Titre     :   Verification de la non nullite de l'operande/resultat RE2
;               ---------------------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   C=0 si RE2 ne contient pas +0 ni -0 et uniquement ces cas,
;               sinon C=1 (RE2 reste inchange dans tous les cas)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
anl     A,#7fh
jnz     F32ZeroRE2End
mov     A,RE22                  ; RE23=00h ou 80h
jnz     F32ZeroRE2End
mov     A,RE21                  ; RE22=00h
jnz     F32ZeroRE2End
mov     A,RE20                  ; RE21=00h
jnz     F32ZeroRE2End
setb    C                       ; RE20=00h donc +0 ou -0
ret
F32ZeroRE2End:
clr     C
ret

F32Cvti32tof32RE1:
;
; Titre     :   Conversion d'un entier signe de 32 bits vers un flottant de 32 bits dans RE1
;               ----------------------------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie en flottant dans RE1 avec C=0
; Utilise   :   Registres A, PSW, B, TMP0 et TMP1 non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov		TMP0,#9ch				; 127 + 29 (les decalage pour passer de 1 a 2^29 donc 30 bits)
mov		TMP1,#00h
mov		A,RE13
mov		C,ACC.7
mov		F0,C					; Signe dans F0
jnc		F32Cvti32tof32RE1S
lcall	I32NegRE1
F32Cvti32tof32RE1S:
ljmp	F32NormRE1

F32Cvtf32toi32RE1:
;
; Titre     :   Conversion d'un flottant de 32 bits vers un entier signe de 32 bits dans RE1
;               ----------------------------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 avec C=0 sauf
;				  si RE1<-2^31 sortie de -2^31 avec C=1
;				  si RE1>2^31-1 sortie de 2^31-1 avec C=1
;				Le nombre est arrondi  l'entier inferieur :
;				-1.    -> -1
;				-0.999 -> 0
;				+0.999 -> 0
;				+1.    -> +1
; Utilise   :   Registres A, PSW, B, TMP0 et TMP1 non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
mov		A,RE13
mov		C,ACC.7
mov		F0,C
clr		ACC.7
mov		RE13,A					; Signe dans F0, et valeur absolue du nombre
lcall	F32UCvtf32toi32RE1		; Conversion non signee
jb		F0,F32Cvtf32toi32RE1S3
jnc		F32Cvtf32toi32RE1S1
ret								; Erreur car nombre positf >= 2^32
F32Cvtf32toi32RE1S1:			; Conversion d'un nombre positif sans erreur
mov		A,RE13
jb		ACC.7,F32Cvtf32toi32RE1S2
ret								; Nombre entre 0 et 2^31-1 donc pas d'erreur
F32Cvtf32toi32RE1S2:			; Erreur nombre positif entre 2^31 et 2^32-1
mov		RE13,#7fh
mov		RE12,#ffh
mov		RE11,#ffh
mov		RE10,#ffh
setb	C
ret
F32Cvtf32toi32RE1S3:			; Le nombre est negatif
jnc		F32Cvtf32toi32RE1S5
F32Cvtf32toi32RE1S4:
mov		RE13,#80h				; Erreur car nombre negatif <= -2^32
mov		RE12,#00h
mov		RE11,#00h
mov		RE10,#00h
setb	C
ret
F32Cvtf32toi32RE1S5:			; Conversion d'un nombre negatif sans erreur
mov		A,RE10
subb	A,#01h
mov		A,RE11
subb	A,#00h
mov		A,RE12
subb	A,#00h
mov		A,RE13
subb	A,#80h
jnc		F32Cvtf32toi32RE1S4
lcall	I32NegRE1
clr		C
ret

F32UCvti32tof32RE1:
;
; Titre     :   Conversion d'un entier non signe de 32 bits vers un flottant de 32 bits dans RE1
;               --------------------------------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie en flottant dans RE1 avec C=0
; Utilise   :   Registres A, PSW, B, TMP0 et TMP1 non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov		TMP0,#9ch				; 127 + 29 (les decalage pour passer de 1 a 2^29 donc 30 bits)
mov		TMP1,#00h
clr		F0						; Nombre positif
ljmp	F32NormRE1

F32UCvtf32toi32RE1:
;
; Titre     :   Conversion d'un flottant de 32 bits vers un entier non signe de 32 bits dans RE1
;               --------------------------------------------------------------------------------
;
; Entree(s) :   Donnee dans RE10
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 avec C=0 sauf
;				  si RE1<=-0.5   sortie de 0 avec C=1
;				  si RE1>+2^32-1 sortie de 2^32-1 avec C=1
;				Le nombre est arrondi  l'entier inferieur :
;				-0.499 -> 0
;				+0.999 -> 0
;				+1.    -> +1
; Utilise   :   Registres A, PSW, B, TMP0 et TMP1 non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov		A,RE13
jnb		ACC.7,F32UCvtf32toi32RE1S2
clr		C
subb	A,#bfh
jc		F32UCvtf32toi32RE1S1
mov		RE13,#0					; Nombre inferieur ou egal a -0,5 donc erreur
mov		RE12,#0
mov		RE11,#0
mov		RE10,#0
setb	C
ret
F32UCvtf32toi32RE1S1:			; Nombre negatif superieur a -0,5 donc 0 sans erreur
mov		RE13,#0
mov		RE12,#0
mov		RE11,#0
mov		RE10,#0
clr		C
ret
F32UCvtf32toi32RE1S2:			; Nombre positif
clr		C
mov		A,RE12
subb	A,#80h
mov		A,RE13
subb	A,#4fh
jc		F32UCvtf32toi32RE1S3
mov		RE13,#ffh				; Nombre superieur ou egal a 2^32 donc erreur
mov		RE12,#ffh
mov		RE11,#ffh
mov		RE10,#ffh
setb	C
ret
F32UCvtf32toi32RE1S3:			; Nombre positif entre 0 et 2^32-1
clr		C
mov		A,RE12
subb	A,#7fh
mov		A,RE13
subb	A,#3fh
jc		F32UCvtf32toi32RE1S1	; Nombre entre 0 et 1. (exclu)
lcall	F32GetexpRE1			; Nombre entre 1 et 2^32-1
mov		TMP0,A
mov		RE13,#00
mov		A,RE12
setb	ACC.7
mov		RE12,A					; Mantisse dans RE1 exposant dans TMP0
clr		C
mov		A,TMP0
subb	A,#96h
jc		F32UCvtf32toi32RE1S4
lcall	I32ShLeRE1				; Nombre entre 2^24 et 2^32 donc shift gauche
clr		C
ret
F32UCvtf32toi32RE1S4:			; Nombre ebtre 2^24-1 et 2^32 donc shift droite
cpl		A
inc		A
lcall	I32ShRiRE1
clr		C
ret

;
;   Sous-programmes internes
;

F32GetexpRE1:
;
; Titre     :   Decomposition de l'argument RE1
;               -------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Exposant dans A
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE12
rlc     A
mov     A,RE13
rlc     A
ret

F32GetexpRE2:
;
; Titre     :   Decomposition de l'argument RE2
;               -------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   Exposant dans A
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE22
rlc     A
mov     A,RE23
rlc     A
ret

F32RE22RE1:
;
; Titre     :   Copie de l'argument RE2 vers l'argument RE1
;               -------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   RE1 contenant la valeur de RE2
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE13,RE23
mov     RE12,RE22
mov     RE11,RE21
mov     RE10,RE20
ret

F32NormRE1:
;
; Titre     :   Normalisation d'un nombre dans RE1 et sortie
;               --------------------------------------------
;
; Entree(s) :   RE1 la mantisse (non signee) sur un nombre de bits variable
;				F0 signe du nombre a sortir
;				TMP1,TMP0 l'exposant sur 16 bits correspondant a une
;				mantisse sur 30 bits (codage IEEE754) donc si elle est de
;				la forme : 001xxxxx xxxxxxxx xxxxxxxx xxxxxxxx
;				(mantisse IEEE754 multipliee par 64)
;				Exemple : TMP1,TMP0=127 mantisse=00100000 00000000 0000000 00000000
;				En sortie on aura 3f 80 00 00 soit 1.
; Sortie(s) :   RE1 normalis (dnormalis, infini ou nul) selon IEEE754
; Utilise   :   Registres A, PSW, B, TMP0 et TMP1 non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov		A,RE13
jnz		F32NormRE1B1
mov		A,RE12
jnz		F32NormRE1B1
mov		A,RE11
jnz		F32NormRE1B1
mov		A,RE10
jnz		F32NormRE1B1
mov		C,F0					; Nombre nul a sortir (+ ou - 0)
clr		A
rrc		A
ret
F32NormRE1B1:					; Nombre non nul, normalisation vers le haut
mov     A,RE13
anl		A,#11100000b
jnz     F32NormRE1B2
mov     A,#1                    ; Le nombre tient sur moins de 30 bits
lcall   I32ShLeRE1              ; donc decalage a gauche
clr		C
mov		A,TMP0
subb	A,#1
mov		TMP0,A
mov		A,TMP1
subb	A,#0
mov		TMP1,A
sjmp    F32NormRE1B1
F32NormRE1B2:					; Mantisse de 30 a 32 bits, normalisation vers le bas
mov     A,RE13                  ; Normalisation vers le bas
anl		A,#11000000b
jz      F32NormRE1S1
mov     A,#1                    ; Le nombre tient sur plus de 30 bits
lcall   I32ShRiRE1              ; donc decalage a droite
mov		A,TMP0
add		A,#1
mov		TMP0,A
mov		A,TMP1
addc	A,#0
mov		TMP1,A
sjmp	F32NormRE1B2
F32NormRE1S1:					; Mantisse sur exactement 30 bits
mov		A,#20h					; Arrondi de la mantisse
add		A,RE10
mov		RE10,A
mov		A,#0
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A					; Mantisse arrondi sur 30 ou 31 bits
anl		A,#11000000b
jz		F32NormRE1S2
mov		A,#20h					; L'arrondi a fait deborder la mantisse de 30 bits
add		A,RE10					; donc arrondi *2 et decalage a droite d'un cran
mov		RE10,A
mov		A,#0
addc	A,RE11
mov		RE11,A
mov		A,#0
addc	A,RE12
mov		RE12,A
mov		A,#0
addc	A,RE13
mov		RE13,A
mov		A,#1
lcall	I32ShRiRE1
mov		A,TMP0
add		A,#1
mov		TMP0,A
mov		A,TMP1
addc	A,#0
mov		TMP1,A
F32NormRE1S2:					; Mantisse arrondie sur exactement 30 bits
mov		A,TMP1
jb		ACC.7,F32NormRE1S5		; Si exposant negatif
jnz		F32NormRE1S3			; Si exposant positif et >=256
mov		A,TMP0					; Exposant positif et <256
jz		F32NormRE1S6			; Si exposant nul
cjne	A,#ffh,F32NormRE1S4
F32NormRE1S3:					; Overflow car exposant >=255
F32OutInf:
;
; Titre     :   Sortie de + ou moins l'infini
;               -----------------------------
;
; Entree(s) :   Signe de l'infini a sortir dans F0
; Sortie(s) :   RE1 contenant + ou - l'infini
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;						
mov		A,#7fh
mov		C,F0
mov		ACC.7,C 
mov		RE13,A
mov		RE12,#80h
mov		RE11,#00h
mov		RE10,#00h
setb	C
ret
F32NormRE1S4:					; Exposant positif non nul et <255
mov		A,#6					; Rangement et sortie
lcall	I32ShRiRE1
mov		A,TMP0	
mov		C,F0
rrc		A
mov		RE13,A
mov		A,RE12
mov		ACC.7,C
mov		RE12,A
clr		C
ret
F32NormRE1S5:					; Exposant negatif
cjne	A,#ffh,F32NormRE1S7		; Si exposant <-256
mov		A,TMP0					; Exposant de -1 a -256
jnb		ACC.7,F32NormRE1S7		; Si exposant entre -129 et -256
F32NormRE1S6:					; Exposant entre 0 et -128
mov		A,TMP0
cpl		A
add		A,#8
lcall	I32ShRiRE1				; Decalage a droite
mov		A,#0
mov		C,F0
mov		ACC.7,C
mov		RE13,A
clr		C
ret
F32NormRE1S7:					; Sortie de + ou - 0
mov		RE10,#0
mov		RE11,#0
mov		RE12,#0
mov		A,#0
mov		C,F0
mov		ACC.7,C
mov		RE13,A
clr		C
ret
