; LIBI08.ASM

;       Bibliotheque mathematique pour les entiers sur 08 bits
;       ------------------------------------------------------
;
;                     (C) E+E CHAMOUARD 2001

; La bibliotheque offre les sous-programmes mathematiques pour les nombres
;   fixes codes sur 08 bits en complement a deux pour les nombres signes
;   et en binaire pur pour les nombres non signes.

; Elle utilise les memes conventions que les autres bibliotheques mathematiques
;   (code de retour en cas d'erreur,...) et (obligatoirement) les memes registres.
; C'est son unique raison d'etre car les operations fournies sont (presque)
;   triviales sur une architecture MCS51.

; La bibliotheque travaille sur 2 groupes de registres :
;   RE10 pour le premier operande ou premier resultat,
;   RE20 pour le second operande ou second resultat.

; Le programme principal doit donc definir 2 registres directs :
;   direct  RE10, xxh   ; Premier operande ou premier resultat
;   direct  RE20, xxh   ; Second operande ou second resultat

; Le chargement ou sauvegarde des resultats ou des operandes s'effectue
;   par chargement  :
;       dans la memoire externe   (ex : I08ReadExtREx),
;       dans la memoire programme (ex : I08ReadProgREx),
;       dans la memoire interne   (ex : I08ReadIntREx).

; Les operations s'effectuent en signe (ex I08Mul) ou en non signe
;   (ex I08UMul).

; La carry (C) est positionnee en cas de debordement ou d'erreur (division)
;   pour tous les sous-programmes sauf ceux de chargement ou de sauvegarde.

; Liste des sous-programmes :
;   I08ReadExtRE1               Lecture de RE1 depuis la memoire externe
;   I08ReadExtRE2               Lecture de RE2 depuis la memoire externe
;   I08ReadProgRE1              Lecture de RE1 depuis la memoire programme
;   I08ReadProgRE2              Lecture de RE2 depuis la memoire programme
;   I08ReadIntRE1               Lecture de RE1 depuis la memoire interne
;   I08ReadIntRE2               Lecture de RE2 depuis la memoire interne
;   I08WriteExtRE1              Ecriture de RE1 dans la memoire externe
;   I08WriteExtRE2              Ecriture de RE2 dans la memoire externe
;   I08WriteIntRE1              Ecriture de RE1 dans la memoire interne
;   I08WriteIntRE2              Ecriture de RE2 dans la memoire interne
;   I08ClearRE1                 Mise a 0 de RE1
;   I08UAdd                     RE1=RE1+RE2 (non signe)
;   I08USub                     RE1=RE1-RE2 (non signe)
;   I08UMul                     RE1=RE1*RE2 (non signe)
;   I08UDiv                     RE1=RE1/RE2 et RE2=RE1%RE2 (non signe)
;   I08ShLeRE1                  RE1=RE1<<A (decalage a gauche non signe)
;   I08ShLeRE2                  RE2=RE2<<A (decalage a gauche non signe)
;   I08ShRiRE1                  RE1=RE1>>A (decalage a droite non signe)
;   I08ShRiRE2                  RE2=RE2>>A (decalage a droite non signe)
;   I08NegRE1                   RE1=-RE1
;   I08SgnRE1                   C=Signe de RE1
;   I08AbsRE1                   RE1=Valeur absolue de RE1
;   I08Add                      RE1=RE1+RE2 (signe)
;   I08Sub                      RE1=RE1-RE2 (signe)
;   I08Mul                      RE1=RE1*RE2 (signe)
;   I08Div                      RE1=RE1/RE2 et RE2=RE1%RE2 (signe)

I08ReadExtRE1:
;
; Titre     :   Lecture de 08 bits de la memoire externe vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   RE1={@DPTR} (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE10,A
inc     DPTR
ret

I08ReadExtRE2:
;
; Titre     :   Lecture de 08 bits de la memoire externe vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   RE2={@DPTR} (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE20,A
inc     DPTR
ret

I08ReadProgRE1:
;
; Titre     :   Lecture de 08 bits de la memoire programme vers l'operande/resultat RE1
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   RE1={@DPTR} (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE10,A
inc     DPTR
ret

I08ReadProgRE2:
;
; Titre     :   Lecture de 08 bits de la memoire programme vers l'operande/resultat RE2
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   RE2={@DPTR} (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE20,A
inc     DPTR
ret

I08ReadIntRE1:
;
; Titre     :   Lecture de 08 bits de la memoire interne vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee
; Sortie(s) :   RE1={@R0} (R0 incremente d'une case)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,@R0
inc     R0
ret

I08ReadIntRE2:
;
; Titre     :   Lecture de 08 bits de la memoire interne vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee
; Sortie(s) :   RE2={@R0} (R0 incremente d"une case)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE20,@R0
inc     R0
ret

I08WriteExtRE1:
;
; Titre     :   Ecriture de 08 bits de l'operande/resultat RE1 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   {@DPTR}=RE1 (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
movx    @DPTR,A
inc     DPTR
ret

I08WriteExtRE2:
;
; Titre     :   Ecriture de 08 bits de l'operande/resultat RE2 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee
; Sortie(s) :   {@DPTR}=RE2 (DPTR incremente d'une case)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
movx    @DPTR,A
inc     DPTR
ret

I08WriteIntRE1:
;
; Titre     :   Ecriture de 08 bits de l'operande/resultat RE1 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee
; Sortie(s) :   {@R0}=RE1 (R0 incremente d'une case)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE10
inc     R0
ret

I08WriteIntRE2:
;
; Titre     :   Ecriture de 08 bits de l'operande/resultat RE2 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee
; Sortie(s) :   {@R0}=RE2 (R0 incremente d'une case)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE20
inc     R0
ret

I08ClearRE1:
;
; Titre     :   RAZ de l'operande/resultat RE1
;               ------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   RE1={0} (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,#0
clr     C
ret

I08UAdd:
;
; Titre     :   Addition non signee de deux nombres de 08 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>ffh), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
ret

I08USub:
;
; Titre     :   Soustraction non signee de deux nombres de 08 bits
;               --------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1<RE2), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
ret

I08UMul:
;
; Titre     :   Multiplication non signee de deux nombres de 08 bits
;               ----------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2 inchange)
;               C=1 si debordement (RE1*RE2>ffh), C=0 sinon
; Utilise   :   Registres A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
mov     B,RE20
mul     AB
mov     RE10,A
mov     C,OV
ret

I08UDiv:
;
; Titre     :   Division non signee de deux nombres de 16 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur (denominateur=0 et dans ce cas RE1=ffh), C=0 sinon
; Utilise   :   Registres A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
jnz     I08UDiv1                ; RE20 nul ?
mov     RE10,#ffh               ; Cas RE20=0 donc erreur
mov     RE20,#0
setb    C
ret
I08UDiv1:                       ; Cas normal
mov     B,A
mov     A,RE10
div     AB
mov     RE10,A
mov     RE20,B
ret

I08ShLeRE1:
;
; Titre     :   Decalage a gauche de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#8,I08ShLeRE1T1
I08ShLeRE1T1:
jc      I08ShLeRE1T2
clr     A                       ; Decalage >=8 donc clear
mov     RE10,A
ret
I08ShLeRE1T2:                   ; Decalage<8
mov     B,A
I08ShLeRE1B:                    ; Boucle sur les decalages
mov     A,B
jnz     I08ShLeRE1S
clr     C
ret
I08ShLeRE1S:
dec     B
clr     C
mov     A,RE10
rlc     A
mov     RE10,A
ljmp    I08ShLeRE1B

I08ShLeRE2:
;
; Titre     :   Decalage a gauche de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#8,I08ShLeRE2T1
I08ShLeRE2T1:
jc      I08ShLeRE2T2
clr     A                       ; Decalage >=8 donc clear
mov     RE20,A
ret
I08ShLeRE2T2:                   ; Decalage<8
mov     B,A
I08ShLeRE2B:                    ; Boucle sur les decalages
mov     A,B
jnz     I08ShLeRE2S
clr     C
ret
I08ShLeRE2S:
dec     B
clr     C
mov     A,RE20
rlc     A
mov     RE20,A
ljmp    I08ShLeRE2B

I08ShRiRE1:
;
; Titre     :   Decalage a droite de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1>>A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#8,I08ShRiRE1T1
I08ShRiRE1T1:
jc      I08ShRiRE1T2
clr     A                       ; Decalage >=8 donc clear
mov     RE10,A
ret
I08ShRiRE1T2:                   ; Decalage<8
mov     B,A
I08ShRiRE1B:                    ; Boucle sur les decalages
mov     A,B
jnz     I08ShRiRE1S
clr     C
ret
I08ShRiRE1S:
dec     B
clr     C
mov     A,RE10
rrc     A
mov     RE10,A
ljmp    I08ShRiRE1B

I08ShRiRE2:
;
; Titre     :   Decalage a droite de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2>>A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#8,I08ShRiRE2T1
I08ShRiRE2T1:
jc      I08ShRiRE2T2
clr     A                       ; Decalage >=8 donc clear
mov     RE20,A
ret
I08ShRiRE2T2:                   ; Decalage<8
mov     B,A
I08ShRiRE2B:                    ; Boucle sur les decalages
mov     A,B
jnz     I08ShRiRE2S
clr     C
ret
I08ShRiRE2S:
dec     B
clr     C
mov     A,RE20
rrc     A
mov     RE20,A
ljmp    I08ShRiRE2B

I08NegRE1:
;
; Titre     :   Changement de signe de l'operande/resultat RE1
;               ----------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=-RE1
;               Avec C=1 si debordement (si RE1=80h=-128) et C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
clr     C
xrl     A,#80h
jnz     I08NegRE1F              ; Test si RE10=80h
setb    C                       ; C=1 car RE1=80h (cas de debordement)
I08NegRE1F:
ret

I08SgnRE1:
;
; Titre     :   Recuperation du signe de l'operande/resultat RE1
;               ------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si le nombre est positif ou nul
;               C=1 si le nombre est negatif
;               RE1 reste inchange
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
mov     C,ACC.7
ret

I08AbsRE1:
;
; Titre     :   Valeur absolue de l'operande/resultat RE1
;               -----------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=ABS(RE1)
;               C=1 si debordement en codage signe (RE1=80h=-128)
;               C=0 si nombre compatible d'un codage signe
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   I08SgnRE1
jnc     I08AbsRE1F
ljmp    I08NegRE1               ; Change le signe du nombre negatif
I08AbsRE1F:                     ; Rien a faire car nombre positif
ret

I08Add:
;
; Titre     :   Addition signee de deux nombres de 08 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>7fh ou <80h), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
mov     C,OV
ret

I08Sub:
;
; Titre     :   Soustraction signee de deux nombres de 08 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1-RE2>7fh ou <80h), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
mov     C,OV
ret

I08Mul:
;
; Titre     :   Multiplication signee de deux nombres de 08 bits
;               ------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2 inchange)
;               C=1 si debordement (RE1*RE2>7fh ou <80h), C=0 sinon
; Utilise   :   Registres A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
jb      ACC.7,I08MulS
mov     B,A                     ; RE10 positif
mov     A,RE20
jnb     ACC.7,I08MulPositif
cpl     A                       ; RE20 negatif donc change de signe
add     A,#1
I08MulNegatif:                  ; Le resultat est negatif
mul     AB
jb      OV,I08MulE2             ; Verification pas de depassement de 255
jz      I08MulNul               ; Cas d'un resultat nul (0*(-qq chose))
cpl     A
add     A,#1
mov     RE10,A
jnb     ACC.7,I08MulE           ; Verification pas de depassement de -128
clr     C
ret
I08MulS:
cpl     A                       ; RE10 negatif donc change de signe
add     A,#1
mov     B,A
mov     A,RE20
jnb     ACC.7,I08MulNegatif
cpl     A                       ; RE20 negatif donc change de signe
add     A,#1
I08MulPositif:                  ; Le resultat est positif
mul     AB
mov     RE10,A
jb      OV,I08MulE              ; Verification pas de depassement de 255
jb      ACC.7,I08MulE           ; Verification pas de depassement de 127
clr     C
ret
I08MulNul:                      ; Resultat nul
mov     RE10,A
clr     C
ret
I08MulE2:
cpl     A
add     A,#1
mov     RE10,A
I08MulE:
setb    C
ret

I08Div:
;
; Titre     :   Division signee de deux nombres de 08 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur : - denominateur=0 et dans ce cas RE1=7fh ou 80h
;                                 selon le signe de RE10)
;                               - debordement du quotien en signe (+128)
;               C=0 si pas d'erreur
; Utilise   :   Registres A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
jnz     I08Div1                 ; RE20 nul ?
mov     RE20,#0
mov     A,RE10
jb      ACC.7,I08DivE1
mov     RE10,#7fh               ; Cas RE20=0 et RE10>=0
setb    C
ret
I08DivE1:                       ; Cas RE20=0 et RE10<0
mov     RE10,#80h
setb    C
ret
I08Div1:                        ; Cas normal RE20!=0
jb      ACC.7,I08DivS1
mov     B,A                     ; RE20>0
mov     A,RE10
jb      ACC.7,I08DivS2
div     AB                      ; RE20>0 et RE10>=0
mov     RE10,A
mov     RE20,B
ret
I08DivS2:                       ; RE20>0 et RE10<0
cpl     A
inc     A                       ; Changement de signe de RE10
I08DivS3:
div     AB
cpl     A
inc     A                       ; Changement de signe du quotien
mov     RE10,A
mov     A,B
cpl     A
inc     A                       ; Changement de signe du reste
mov     RE20,A
clr     C
ret
I08DivS1:                       ; RE20<0
cpl     A
inc     A                       ; Changement de signe de RE20
mov     B,A
mov     A,RE10
jnb     ACC.7,I08DivS3          ; RE10>=0
cpl     A                       ; RE20<0 et RE10<0
inc     A                       ; Changement de signe de RE10
div     AB
mov     RE10,A
mov     RE20,B
clr     C
jnb     ACC.7,I08DivS4
setb    C                       ; Erreur de debordement : (-128)/(-1)=(+128)
I08DivS4:
ret
