; LIBI16.ASM

;       Bibliotheque mathematique pour les entiers sur 16 bits
;       ------------------------------------------------------
;
;                     (C) E+E CHAMOUARD 2001

; La bibliotheque offre les sous-programmes mathematiques pour les nombres
;   fixes codes sur 16 bits en complement a deux pour les nombres signes
;   et en binaire pur pour les nombres non signes.

; Elle utilise les memes conventions que les autres bibliotheques mathematiques
;   (code de retour en cas d'erreur,...) et (obligatoirement) les memes registres.

; La bibliotheque travaille sur 3 groupes de registres :
;   RE1x pour le premier operande ou premier resultat,
;   RE2x pour le second operande ou second resultat,
;   TMPx pour le registre temporaire.

; Le programme principal doit donc definir 6 registres directs :
;   direct  RE10, xxh   ; LSB du premier operande ou premier resultat
;   direct  RE11, xxh   ; MSB du premier operande ou premier resultat
;   direct  RE20, xxh   ; LSB du second operande ou second resultat
;   direct  RE21, xxh   ; MSB du second operande ou second resultat
;   direct  TMP0, xxh   ; LSB du resultat intermediaire
;   direct  TMP1, xxh   ; MSB du resultat intermediaire

; Le chargement ou sauvegarde des resultats ou des operandes s'effectue
;   par chargement  :
;       dans la memoire externe   (ex : I16ReadExtREx),
;       dans la memoire programme (ex : I16ReadProgREx),
;       dans la memoire interne   (ex : I16ReadIntREx).
; L'octet de poid faible est stocke en tete (avec l'adresse la plus basse).

; Les operations s'effectuent en signe (ex I16Mul) ou en non signe
;   (ex I16UMul).

; La carry (C) est positionnee en cas de debordement ou d'erreur (division)
;   pour tous les sous-programmes sauf ceux de chargement ou de sauvegarde.

; Liste des sous-programmes :
;   I16ReadExtRE1               Lecture de RE1 depuis la memoire externe
;   I16ReadExtRE2               Lecture de RE2 depuis la memoire externe
;   I16ReadProgRE1              Lecture de RE1 depuis la memoire programme
;   I16ReadProgRE2              Lecture de RE2 depuis la memoire programme
;   I16ReadIntRE1               Lecture de RE1 depuis la memoire interne
;   I16ReadIntRE2               Lecture de RE2 depuis la memoire interne
;   I16WriteExtRE1              Ecriture de RE1 dans la memoire externe
;   I16WriteExtRE2              Ecriture de RE2 dans la memoire externe
;   I16WriteIntRE1              Ecriture de RE1 dans la memoire interne
;   I16WriteIntRE2              Ecriture de RE2 dans la memoire interne
;   I16ClearRE1                 Mise a 0 de RE1
;   I16UAdd                     RE1=RE1+RE2 (non signe)
;   I16USub                     RE1=RE1-RE2 (non signe)
;   I16UMul                     RE1=RE1*RE2 (non signe)
;   I16UDiv                     RE1=RE1/RE2 et RE2=RE1%RE2 (non signe)
;   I16ShLeRE1                  RE1=RE1<<A (decalage a gauche non signe)
;   I16ShLeRE2                  RE2=RE2<<A (decalage a gauche non signe)
;   I16ShRiRE1                  RE1=RE1>>A (decalage a droite non signe)
;   I16ShRiRE2                  RE2=RE2>>A (decalage a droite non signe)
;   I16NegRE1                   RE1=-RE1
;   I16SgnRE1                   C=Signe de RE1
;   I16AbsRE1                   RE1=Valeur absolue de RE1
;   I16Add                      RE1=RE1+RE2 (signe)
;   I16Sub                      RE1=RE1-RE2 (signe)
;   I16Mul                      RE1=RE1*RE2 (signe)
;   I16Div                      RE1=RE1/RE2 et RE2=RE1%RE2 (signe)
;   I16Cvti16toi08RE1           RE1 converti de 16 a 8 bits en signe
;   I16Cvti08toi16RE1           RE1 converti de 8 a 16 bits en signe
;   I16UCvti16toi08RE1          RE1 converti de 16 a 8 bits en non signe
;   I16UCvti08toi16RE1          RE1 converti de 8 a 16 bits en non signe

I16ReadExtRE1:
;
; Titre     :   Lecture de 16 bits de la memoire externe vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+1,@DPTR} (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE10,A
inc     DPTR
movx    A,@DPTR
mov     RE11,A
inc     DPTR
ret

I16ReadExtRE2:
;
; Titre     :   Lecture de 16 bits de la memoire externe vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+1,@DPTR} (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE20,A
inc     DPTR
movx    A,@DPTR
mov     RE21,A
inc     DPTR
ret

I16ReadProgRE1:
;
; Titre     :   Lecture de 16 bits de la memoire programme vers l'operande/resultat RE1
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+1,@DPTR} (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE10,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE11,A
inc     DPTR
ret

I16ReadProgRE2:
;
; Titre     :   Lecture de 16 bits de la memoire programme vers l'operande/resultat RE2
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+1,@DPTR} (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE20,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE21,A
inc     DPTR
ret

I16ReadIntRE1:
;
; Titre     :   Lecture de 16 bits de la memoire interne vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@R0+1,@R0} (R0 incremente de deux cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,@R0
inc     R0
mov     RE11,@R0
inc     R0
ret

I16ReadIntRE2:
;
; Titre     :   Lecture de 16 bits de la memoire interne vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@R0+1,@R0} (R0 incremente de deux cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE20,@R0
inc     R0
mov     RE21,@R0
inc     R0
ret

I16WriteExtRE1:
;
; Titre     :   Ecriture de 16 bits de l'operande/resultat RE1 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+1,@DPTR}=RE1 (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
movx    @DPTR,A
inc     DPTR
mov     A,RE11
movx    @DPTR,A
inc     DPTR
ret

I16WriteExtRE2:
;
; Titre     :   Ecriture de 16 bits de l'operande/resultat RE2 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+1,@DPTR}=RE2 (DPTR incremente de deux cases)
; Utilise   :   Registres A et PSW non souvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
movx    @DPTR,A
inc     DPTR
mov     A,RE21
movx    @DPTR,A
inc     DPTR
ret

I16WriteIntRE1:
;
; Titre     :   Ecriture de 16 bits de l'operande/resultat RE1 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+1,@R0}=RE1 (R0 incremente de deux cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE10
inc     R0
mov     @R0,RE11
inc     R0
ret

I16WriteIntRE2:
;
; Titre     :   Ecriture de 16 bits de l'operande/resultat RE2 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+1,@R0}=RE2 (R0 incremente de deux cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE20
inc     R0
mov     @R0,RE21
inc     R0
ret

I16ClearRE1:
;
; Titre     :   RAZ de l'operande/resultat RE1
;               ------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   RE1={0,0} (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,#0
mov     RE11,#0
clr     C
ret

I16UAdd:
;
; Titre     :   Addition non signee de deux nombres de 16 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>ffffh), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
mov     A,RE11
addc    A,RE21
mov     RE11,A
ret

I16USub:
;
; Titre     :   Soustraction non signee de deux nombres de 16 bits
;               --------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1<RE2), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
mov     A,RE11
subb    A,RE21
mov     RE11,A
ret

I16UMul:
;
; Titre     :   Multiplication non signee de deux nombres de 16 bits
;               ----------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2 inchange)
;               C=1 si debordement (RE1*RE2>ffffh), C=0 sinon
; Utilise   :   Registres TMP0, TMP1, A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
mov     TMP0,A
mov     TMP1,RE11
mov     B,RE20
mul     AB
mov     RE10,A
mov     RE11,B                  ; Premier produit stocke
mov     A,TMP1
mov     B,RE20
mul     AB
add     A,RE11
mov     RE11,A                  ; Second produit stocke
mov     F0,C                    ; Test si debordement dans l'addition
mov     A,B
jz      I16UMul1                ; Test si debordement dans le second produit
setb    F0
I16UMul1:
mov     A,TMP0
mov     B,RE21
mul     AB
add     A,RE11
mov     RE11,A                  ; Troisieme produit stocke
orl     C,F0
mov     F0,C                    ; Test si debordement dans l'addition
mov     A,B
jz      I16UMul2                ; Test si debordement dans le troisieme produit
setb    F0
I16UMul2:
mov     A,TMP1
jz      I16UMul3
mov     A,RE21
jz      I16UMul3                ; Test si quatrieme produit nul
setb    F0
I16UMul3:
mov     C,F0
ret

I16UDiv:
;
; Titre     :   Division non signee de deux nombres de 16 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur (denominateur=0 et dans ce cas RE1=ffffh), C=0 sinon
; Utilise   :   Registres TMP1, TMP0, A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE21
jnz     I16UDiv1                ; RE21 nul ?
mov     A,RE20                  ; RE21=0
jnz     I16UDiv1                ; RE20 nul ?
mov     RE11,#ffh               ; Cas RE21=RE20=0 donc erreur
mov     RE10,#ffh
mov     RE21,#0
mov     RE20,#0
setb    C
ret
I16UDiv1:                       ; Cas normal
clr     A
xch     A,RE20
mov     TMP0,A
clr     A
xch     A,RE21
mov     TMP1,A                  ; TMP=Denominateur, RE1=Numerateur et RE2=0
mov     B,#16
I16UDiv1B:                      ; Boucle de calcul
mov     A,RE10
add     A,RE10
mov     RE10,A
mov     A,RE11
rlc     A
mov     RE11,A                  ; RE1=2*RE1
mov     A,RE20
rlc     A
mov     RE20,A
mov     A,RE21
rlc     A
mov     RE21,A                  ; RE2=2*RE2
mov     A,RE20
subb    A,TMP0
mov     A,RE21
subb    A,TMP1
jc      I16UDiv1S               ; RE2<TMP ?
mov     RE21,A                  ; RE2>=TMP
mov     A,RE20
subb    A,TMP0
mov     RE20,A                  ; RE2=RE2-TMP
inc     RE10
I16UDiv1S:
djnz    B,I16UDiv1B
clr     C
ret

I16ShLeRE1:
;
; Titre     :   Decalage a gauche de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#16,I16ShLeRE1T1
I16ShLeRE1T1:
jc      I16ShLeRE1T2
clr     A                       ; Decalage >=16 donc clear
mov     RE11,A
mov     RE10,A
ret
I16ShLeRE1T2:                   ; Decalage<16
mov     B,A
jnb     B.3,I16ShLeRE1B
mov     RE11,RE10               ; Decalage>=8 donc shift 8 bits en bloc
clr     A
mov     RE10,A
clr     B.3
I16ShLeRE1B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I16ShLeRE1S
clr     C
ret
I16ShLeRE1S:
dec     B
clr     C
mov     A,RE10
rlc     A
mov     RE10,A
mov     A,RE11
rlc     A
mov     RE11,A
ljmp    I16ShLeRE1B

I16ShLeRE2:
; Titre     :   Decalage a gauche de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#16,I16ShLeRE2T1
I16ShLeRE2T1:
jc      I16ShLeRE2T2
clr     A                       ; Decalage >=16 donc clear
mov     RE21,A
mov     RE20,A
ret
I16ShLeRE2T2:                   ; Decalage<16
mov     B,A
jnb     B.3,I16ShLeRE2B
mov     RE21,RE20               ; Decalage>=8 donc shift 8 bits en bloc
clr     A
mov     RE20,A
clr     B.3
I16ShLeRE2B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I16ShLeRE2S
clr     C
ret
I16ShLeRE2S:
dec     B
clr     C
mov     A,RE20
rlc     A
mov     RE20,A
mov     A,RE21
rlc     A
mov     RE21,A
ljmp    I16ShLeRE2B

I16ShRiRE1:
;
; Titre     :   Decalage a droite de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1>>A (decalage a droite non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#16,I16ShRiRE1T1
I16ShRiRE1T1:
jc      I16ShRiRE1T2
clr     A                       ; Decalage >=16 donc clear
mov     RE11,A
mov     RE10,A
ret
I16ShRiRE1T2:                   ; Decalage<16
mov     B,A
jnb     B.3,I16ShRiRE1B
mov     RE10,RE11               ; Decalage>=8 donc shift 8 bits en bloc
clr     A
mov     RE11,A
clr     B.3
I16ShRiRE1B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I16ShRiRE1S
clr     C
ret
I16ShRiRE1S:
dec     B
clr     C
mov     A,RE11
rrc     A
mov     RE11,A
mov     A,RE10
rrc     A
mov     RE10,A
ljmp    I16ShRiRE1B

I16ShRiRE2:
;
; Titre     :   Decalage a droite de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2>>A (decalage a droite non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#16,I16ShRiRE2T1
I16ShRiRE2T1:
jc      I16ShRiRE2T2
clr     A                       ; Decalage >=16 donc clear
mov     RE21,A
mov     RE20,A
ret
I16ShRiRE2T2:                   ; Decalage<16
mov     B,A
jnb     B.4,I16ShRiRE2B
mov     RE20,RE21               ; Decalage>=8 donc shift 8 bits en bloc
clr     A
mov     RE21,A
clr     B.3
I16ShRiRE2B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I16ShRiRE2S
clr     C
ret
I16ShRiRE2S:
dec     B
clr     C
mov     A,RE21
rrc     A
mov     RE21,A
mov     A,RE20
rrc     A
mov     RE20,A
ljmp    I16ShRiRE2B

I16NegRE1:
;
; Titre     :   Changement de signe de l'operande/resultat RE1
;               ----------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=-RE1
;               Avec C=1 si debordement (si RE1=8000h=-32768) et C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
clr     C
xrl     A,#80h
jnz     I16NegRE1F              ; Test si RE11=80h
mov     A,RE10
jnz     I16NegRE1F              ; Test si RE10=00h
setb    C                       ; C=1 car RE1=8000h (cas de debordement)
I16NegRE1F:
ret

I16SgnRE1:
;
; Titre     :   Recuperation du signe de l'operande/resultat RE1
;               ------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si le nombre est positif
;               C=1 si le nombre est negatif
;               RE1 reste inchange
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE11
mov     C,ACC.7
ret

I16AbsRE1:
;
; Titre     :   Valeur absolue de l'operande/resultat RE1
;               -----------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=ABS(RE1)
;               C=1 si debordement en codage signe (RE1=8000h=-32768)
;               C=0 si nombre compatible d'un codage signe
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   I16SgnRE1
jnc     I16AbsRE1F
ljmp    I16NegRE1               ; Change le signe du nombre negatif
I16AbsRE1F:                     ; Rien a faire car nombre positif
ret

I16Add:
;
; Titre     :   Addition signee de deux nombres de 16 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>7fffh ou <8000h), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
mov     A,RE11
addc    A,RE21
mov     RE11,A
mov     C,OV
ret

I16Sub:
;
; Titre     :   Soustraction signee de deux nombres de 16 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1-RE2>7fffh ou <8000h), C=0 sinon
; Utilise   :   Registres TMP0, TMP1, A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
mov     A,RE11
subb    A,RE21
mov     RE11,A
mov     C,OV
ret

I16Mul:
;
; Titre     :   Multiplication signee de deux nombres de 16 bits
;               ------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2=abs(RE2))
;               C=1 si debordement (RE1*RE2>7fffh ou <8000h), C=0 sinon
; Utilise   :   Registres TMP0, TMP1, A, PSW et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,RE11
jb      ACC.7,I16MulS
mov     A,RE21                  ; RE11 positif
jnb     ACC.7,I16MulPositif
mov     A,RE20                  ; RE20 negatif donc change de signe
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
I16MulNegatif:                  ; Le resultat est negatif
lcall   I16UMul
jc      I16MulE2                ; Verification pas de depassement de 65535
mov     A,RE10
jnz     I16MulNN
mov     A,RE11
jnz     I16MulNN
ret                             ; Cas d'un resultat nul (0*(-qq chose))
I16MulNN:                       ; Resultat non nul
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A                  ; Change le signe du resultat
jnb     ACC.7,I16MulE           ; Verification pas de depassement de -32768
clr     C
ret
I16MulE2:                       ; Erreur avant le changement de signe
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A                  ; Change le signe du resultat
I16MulE:                        ; Erreur
setb    C
ret
I16MulS:                        ; RE10 negatif donc change de signe
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE21
jnb     ACC.7,I16MulNegatif
mov     A,RE20                  ; RE20 negatif donc change de signe
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
I16MulPositif:                  ; Le resultat est positif
lcall   I16UMul
jc      I16MulFin               ; Verification pas de depassement de 65535
mov     A,RE11
mov     C,ACC.7                 ; Donc C=1 si Resultat>=32768
I16MulFin:
ret

I16Div:
;
; Titre     :   Division signee de deux nombres de 16 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur : - denominateur=0 et dans ce cas RE1=7fffh ou 8000h
;                                 selon le signe de RE1)
;                               - debordement du quotien en signe (+32768)
;               C=0 si pas d'erreur
; Utilise   :   Registres TMP1, TMP0, A, PSW et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
jnz     I16Div1                 ; RE21 nul ?
mov     A,RE21
jnz     I16Div1                 ; RE20 nul ?
mov     RE20,#0
mov     RE21,#0
mov     A,RE11
jb      ACC.7,I16DivE1
mov     RE11,#7fh               ; Cas RE2=0 et RE1>=0
mov     RE10,#ffh
setb    C
ret
I16DivE1:                       ; Cas RE2=0 et RE1<0
mov     RE11,#80h
mov     RE10,#00h
setb    C
ret
I16Div1:                        ; Cas normal RE2!=0
mov     A,RE21
jb      ACC.7,I16DivS1
mov     A,RE11                  ; RE2>0
jb      ACC.7,I16DivS2
ljmp    I16UDiv1                ; RE2>0 et RE1>=0
I16DivS2:                       ; RE2>0 et RE1<0
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A                  ; Changement de signe de RE1
I16DivS3:
lcall   I16UDiv1
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A                  ; Changement de signe du quotien
mov     A,RE20
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A                  ; Changement de signe du reste
clr     C
ret
I16DivS1:                       ; RE2<0
mov     A,RE20
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A                  ; Changement de signe de RE2
mov     A,RE11
jnb     ACC.7,I16DivS3          ; RE1>=0
mov     A,RE10                  ; RE2<0 et RE1<0
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A                  ; Changement de signe de RE1
lcall   I16UDiv1
mov     A,RE11
jnb     ACC.7,I16DivS4
setb    C                       ; Erreur de debordement : (-32768)/(-1)=(+32768)
I16DivS4:
ret

I16Cvti16toi08RE1:
;
; Titre     :   Conversion signee de 16 bits vers 8 bits dans RE1
;               -------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 8 bits dans RE10
;               Avec C=1 si RE1>127 ou RE1<-128, C=0 sinon
;               Dans tous les cas RE10=RE1[256] et RE11=0 si >0 RE11=ffh sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE11
jb      ACC.7,I16Cvti16toi08RE1S
mov     A,RE10                  ; Nombre positif
mov     C,ACC.7                 ; Donc C=1 si > 127[256]
mov     A,RE11
mov     RE11,#0
jz      I16Cvti16toi08RE1F
setb    C                       ; C=1 si > 255
I16Cvti16toi08RE1F:
ret
I16Cvti16toi08RE1S:             ; Nombre negatif
mov     A,RE10
mov     C,ACC.7
cpl     C                       ; Donc C=1 si < -128[256]
mov     A,RE11
mov     RE11,#ffh
inc     A
jz      I16Cvti16toi08RE1G
setb    C                       ; C=1 si < -128
I16Cvti16toi08RE1G:
ret

I16Cvti08toi16RE1:
;
; Titre     :   Conversion signee de 8 bits vers 16 bits dans RE1
;               -------------------------------------------------
;
; Entree(s) :   Donnee dans RE10
; Sortie(s) :   Donnee convertie sur 16 bits dans RE1 (C=0)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
clr     C
jb      ACC.7,I16Cvti08toi16RE1S
mov     RE11,#0                 ; Nombre positif
ret
I16Cvti08toi16RE1S:
mov     RE11,#ffh               ; Nombre negatif
ret

I16UCvti16toi08RE1:
;
; Titre     :   Conversion non signee de 16 bits vers 8 bits dans RE1
;               -----------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 8 bits dans RE10 (RE11=0)
;               Avec C=1 si debordement (RE1>255), C=0 sinon
;               Dans tous les cas RE10=RE1[256] et RE11=0
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE11
clr     C
jz      I16UCvti16toi08RE1S
setb    C                       ; C=1 car RE11!=0
I16UCvti16toi08RE1S:
mov     RE11,#0
ret

I16UCvti08toi16RE1:
;
; Titre     :   Conversion non signee de 8 bits vers 16 bits dans RE1
;               -----------------------------------------------------
;
; Entree(s) :   Donnee dans RE10
; Sortie(s) :   Donnee convertie sur 16 bits dans RE1 (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE11,#0
clr     C
ret
