; LIBI2CUS.ASM

;
;    Bibliotheque de sous-programmes d'interface avec des composants I2C
;    -------------------------------------------------------------------
;
;                            (C) E+E CHAMOUARD 2000
;

; Inclure dans le programme principal le gestionnaire I2C :
;   INCLUDE libi2c.asm

; Pour utiliser le PCF8583 :
;   - definir I2CPCF8583Write et I2CPCF8583Read :
;       byte    I2CPCF8583Write,#010100000b
;       byte    I2CPCF8583Read, #010100001b
;   - definir les registres de stockage des informations :
;       direct  RHS, 10h     ; Registre pour les secondes            0 a 59 (en BCD)
;       direct  RHM, 11h     ; Registre pour les minutes             0 a 59 (en BCD)
;       direct  RHH, 12h     ; Registre pour les heures              0 a 23 (en BCD)
;       direct  RHJ, 13h     ; Registre pour les jours (0=lundi)     0 a  6 (en BCD)
;   - si besoin initialiser les interruptions I2C et ecrire la routine de traitement :
;       IE1Trt:              ; Traitement de l'IT I2C
;       push    PSW
;       push    ACC
;       lcall   PCF8583ReadHeure
;       pop     ACC
;       pop     PSW
;       reti
;   - initialiser le PCF8583 au boot avec un lcall PCF8583Init

; Pour utiliser le DS1621 :
;   - definir I2CDS1621Write et I2CDS1621Read :
;       byte    I2CDS1621Write,    #10010000b
;       byte    I2CDS1621Read,     #10010001b
;   - initialiser le DS1621 au boot avec un lcall DS1621Init
;   - pour effectuer une mesure :
;       Appeler DS1621Trigger
;       Attendre la disponibilite de la mesure au besoin en appelant DS1621MesOK
;       Lire la mesure par une DS1621ReadTemp ou DS1621ReadTemp2 (DS1621ReadTemp donne une
;           mesure au demi degre pres alors que DS1621ReadTemp2 permet d'atteindre le dixieme
;           de degre aux temperatures usuelles)

; Cette nouvelle version utilise la compilation conditionnelle, c'est  dire que la definition 
;   de l'adresse d'ecriture I2C du composant provoque la compilation de la partie de la
;   bibliotheque correspondante. Si cette adresse n'est pas definie, les sous-programmes
;   correspondants ne sont pas assembles ce qui evite une inutile surcharge du code
;   operationnel de l'application

;
; Partie PCF8583
;

IF I2CPCF8583Write

PCF8583Init:
;
; Titre     :   Init du PCF8583
;               ---------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583InitErreur
mov     A,#00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583InitErreur
mov     A,#00h                  ; Status register = 00 h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583InitErreur
lcall   I2CStop
clr     C
ret
PCF8583InitErreur:
lcall   I2CStop
setb    C
ret

PCF8583WriteHeure:
;
; Titre     :   Ecriture de l'heure sur le PCF8583
;               ----------------------------------
;
; Entree(s) :   RHS : secondes en BCD
;               RHM : minutes en BCD
;               RHH : heures en BCD
;               RHJ : jour (0=lundi)
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#80h                  ; Status register = 80h (stop durant la prog)
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h                  ; Centiemes de seconde = 00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h                  ; Secondes = 00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,RHM                   ; Minutes = RHM
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,RHH                   ; Heures = RHH
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h                  ; Annee/data = 00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,RHJ                   ; Jour
rl      A
swap    A
anl     A,#11100000b
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
lcall   I2CStop
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
mov     A,#00h                  ; Status register = 00h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteHeureErreur
lcall   I2CStop
clr     C
ret
PCF8583WriteHeureErreur:
lcall   I2CStop
setb    C
ret

PCF8583ReadHeure:
;
; Titre     :   Lecture de l'heure sur le PCF8583
;               ---------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
;               RHS : secondes en BCD
;               RHM : minutes en BCD
;               RHH : heures en BCD
;               RHJ : jour (0=lundi)
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadHeureErreur
mov     A,#02h
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadHeureErreur
lcall   I2CStart
mov     A,#I2CPCF8583Read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadHeureErreur
lcall   I2CReadbyte
mov     RHS,A                   ; Stockage secondes dans RHS
clr     C
lcall   I2CWritebit
lcall   I2CReadbyte
mov     RHM,A                   ; Stockage minutes dans RHM
clr     C
lcall   I2CWritebit
lcall   I2CReadbyte
anl     A,#3fh
mov     RHH,A                   ; Stockage heures dans RHH
clr     C
lcall   I2CWritebit
lcall   I2CReadbyte             ; Lecture sans stockage de l'annee
clr     C
lcall   I2CWritebit
lcall   I2CReadbyte
rr      A
swap    A
anl     A,#07h
mov     RHJ,A                   ; Stockage jour dans RHJ
setb    C
lcall   I2CWritebit
lcall   I2CStop
clr     C
ret
PCF8583ReadHeureErreur:
lcall   I2CStop
setb    C
ret

PCF8583WriteData:
;
; Titre     :   Ecriture d'une donnee dans la RAM du PCF8583
;               --------------------------------------------
;
; Entree(s) :   Donnee a ecrire dans ACC
;               Adresse d'ecriture dans B (valide de 10h a ffh)
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes (B inchange)
; Pile      :   7 avec l'appel
; Cycles    :   ?
;
push    ACC
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteDataErreur1
mov     A,B
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteDataErreur1
pop     ACC
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583WriteDataErreur2
lcall   I2CStop
clr     C
ret
PCF8583WriteDataErreur1:
pop     ACC
PCF8583WriteDataErreur2:
lcall   I2CStop
setb    C
ret

PCF8583ReadData:
;
; Titre     :   Lecture d'une donnee dans la RAM du PCF8583
;               -------------------------------------------
;
; Entree(s) :   Adresse d'ecriture dans B (valide de 10h a ffh)
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
;               Donnee lue dans A
; Utilise   :   ACC et PSW non sauvegardes (B inchange)
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CPCF8583Write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadDataErreur
mov     A,B
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadDataErreur
lcall   I2CStart
mov     A,#I2CPCF8583Read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      PCF8583ReadDataErreur
lcall   I2CReadbyte
setb    C
lcall   I2CWritebit
lcall   I2CStop
clr     C
ret
PCF8583ReadDataErreur:
lcall   I2CStop
setb    C
ret

ENDIF

;
; Partie DS1621
;

IF I2CDS1621Write

DS1621Init:
;
; Titre     :   Init du DS1621
;               --------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   8 avec l'appel
; Cycles    :   ?
;
lcall   DS1621ReadStatus        ; Lecture de l'octet de configuration (au cas ou il serait bon)
jc      DS1621InitErreur        ; Sortie si erreur I2C
anl     A,#00000011b            ; Verification de l'octet
cjne    A,#00000001b,DS1621InitWithWrite
clr     C                       ; Sortie car config OK
ret
DS1621InitWithWrite:            ; Positionnement du configuration byte
mov     ACC,#00000001b          ; pop=0 1shot=1
ljmp    DS1621WriteStatus
DS1621InitErreur:
setb    C
ret

DS1621Trigger:
;
; Titre     :   Declenche une mesure de temperature par le DS1621
;               -------------------------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621WriteTriggerErreur
mov     A,#EEh                  ; Start conversion command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621WriteTriggerErreur
lcall   I2CStop
clr     C
ret
DS1621WriteTriggerErreur:
lcall   I2CStop
setb    C
ret

DS1621MesOK:
;
; Titre     :   Verifie si mesure DS1621 prete
;               ------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si mesure prete et echanges corrects
;               C=1 si mesure pas prete ou pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   8 avec l'appel
; Cycles    :   ?
;
lcall   DS1621ReadStatus
jc      DS1621MesOKErreur
anl     A,#10000000b
cjne    A,#10000000b,DS1621MesOKErreur
clr     C
ret
DS1621MesOKErreur:
setb    C
ret

DS1621ReadTemp:
;
; Titre     :   Lecture de la temperature sur le DS1621
;               ---------------------------------------
;
; Entree(s) :   Rien (pas de verification d'une fin de conversion !)
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
;               B :   octet haut ( C en binaire)
;               ACC : octet bas  (fractions de  en binaire)
;               ACC=0 ou 80h (resolution d'un demi )
; Utilise   :   ACC, B et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTempErreur
mov     A,#AAh                  ; Read temperature command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTempErreur
lcall   I2CStart
mov     A,#I2CDS1621Read        ; DS1621 read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTempErreur
lcall   I2CReadbyte
mov     B,A                     ; Stockage octet haut dans B
clr     C
lcall   I2CWritebit
lcall   I2CReadbyte             ; Octet bas dans ACC
setb    C
lcall   I2CWritebit
lcall   I2CStop
clr     C
ret
DS1621ReadTempErreur:
lcall   I2CStop
setb    C
ret

DS1621ReadTemp2:
;
; Titre     :   Lecture de la temperature sur le DS1621 version 2
;               -------------------------------------------------
;
; Entree(s) :   Rien (pas de verification d'une fin de conversion !)
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
;               B :   octet haut ( C en binaire)
;               ACC : octet bas  (fractions de  en binaire)
;               Cette version calcule la temperature avec plus de
;               precision (environ le dixieme de  a T "normale")
; Utilise   :   ACC, B et PSW non sauvegardes
;               DPH et DPL sauvegardes
; Pile      :   8 avec l'appel
; Cycles    :   ?
;
push    DPH
push    DPL
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
mov     A,#AAh                  ; Read temperature command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CStart
mov     A,#I2CDS1621Read        ; DS1621 read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CReadbyte
mov     DPH,A                   ; Stockage octet haut dans DPH
dec     DPH                     ; Temp=Temp-1
setb    C
lcall   I2CWritebit
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
mov     A,#A9h                  ; Read slope command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CStart
mov     A,#I2CDS1621Read        ; DS1621 read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CReadbyte
mov     B,A                     ; Stockage slope dans B
setb    C
lcall   I2CWritebit
ljmp    DS1621ReadTemp2Suite
DS1621ReadTemp2Erreur:          ; Sortie sur erreur
lcall   I2CStop
pop     DPL
pop     DPH
setb    C
ret
DS1621ReadTemp2Suite:
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
mov     A,#A8h                  ; Read counter command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CStart
mov     A,#I2CDS1621Read        ; DS1621 read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadTemp2Erreur
lcall   I2CReadbyte             ; Counter dans A
setb    C
lcall   I2CWritebit
lcall   I2CStop
;push    ACC
;lcall   ComWriteByteAscii
;mov     A,B
;lcall   ComWriteByteAscii
;pop     ACC
cpl     A
add     A,B
inc     A                       ; A=Slope-Counter B=Slope
push    DPH
mov     DPH,#8
mov     DPL,#0                  ; A=Numerateur, DPH=8, DPL=Quotien, B=Denominateur
DS1621ReadTemp2Bits:
clr     C
rlc     A                       ; A=2*A
xch     A,DPL
rlc     A
xch     A,DPL                   ; DPL=2*DPL
cjne    A,B,DS1621ReadTemp2Nope
DS1621ReadTemp2Nope:
jc      DS1621ReadTemp2Next
subb    A,B
inc     DPL
DS1621ReadTemp2Next:
djnz    DPH,DS1621ReadTemp2Bits ; Boucle sur les 8 bits
pop     DPH
DS1621ReadTemp2End:             ; Ajoute 0,75 a DPHL et sort
mov     A,#192
add     A,DPL
mov     DPL,A
mov     A,#0
addc    A,DPH
mov     B,A
mov     A,DPL
pop     DPL
pop     DPH
clr     C
ret

DS1621ReadStatus:
;
; Titre     :   Read status du DS1621
;               ---------------------
;
; Entree(s) :   Rien
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
;               ACC=status (octet de configuration)
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadStatusErreur
mov     A,#ACh                  ; Read configuration command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadStatusErreur
lcall   I2CStart
mov     A,#I2CDS1621Read        ; DS1621 read
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621ReadStatusErreur
lcall   I2CReadbyte             ; Read configuration byte
setb    C                       ; Master NACK and end
lcall   I2CWritebit
lcall   I2CStop
clr     C
ret
DS1621ReadStatusErreur:
lcall   I2CStop
setb    C
ret

DS1621WriteStatus:
;
; Titre     :   Write status du DS1621
;               ----------------------
;
; Entree(s) :   ACC=Nouveau status
; Sortie(s) :   C=0 si echanges corrects
;               C=1 si pas d'acquitement I2C
; Utilise   :   ACC et PSW non sauvegardes
; Pile      :   7 avec l'appel
; Cycles    :   ?
;
push    ACC
lcall   I2CStart
mov     A,#I2CDS1621Write       ; DS1621 write
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621WriteStatusErreur1
mov     A,#ACh                  ; Write configuration command
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621WriteStatusErreur1
pop     ACC                     ; Write configuration byte
lcall   I2CWritebyte
lcall   I2CReadbit
jc      DS1621WriteStatusErreur2
lcall   I2CStop
clr     C
ret
DS1621WriteStatusErreur1:
pop     ACC
DS1621WriteStatusErreur2:
lcall   I2CStop
setb    C
ret

ENDIF