; LIBI32.ASM

;       Bibliotheque mathematique pour les entiers sur 32 bits
;       ------------------------------------------------------
;
;                     (C) E+E CHAMOUARD 2001

; La bibliotheque offre les sous-programmes mathematiques pour les nombres
;   fixes codes sur 32 bits en complement a deux pour les nombres signes
;   et en binaire pur pour les nombres non signes.

; Elle utilise les memes conventions que les autres bibliotheques mathematiques
;   (code de retour en cas d'erreur,...) et (obligatoirement) les memes registres.

; La bibliotheque travaille sur 3 groupes de registres :
;   RE1x pour le premier operande ou premier resultat,
;   RE2x pour le second operande ou second resultat,
;   TMPx pour le registre temporaire.

; Le programme principal doit donc definir 12 registres directs :
;   direct  RE10, xxh   ; LSB du premier operande ou premier resultat
;   direct  RE11, xxh   ;
;   direct  RE12, xxh   ;
;   direct  RE13, xxh   ; MSB du premier operande ou premier resultat
;   direct  RE20, xxh   ; LSB du second operande ou second resultat
;   direct  RE21, xxh   ;
;   direct  RE22, xxh   ;
;   direct  RE23, xxh   ; MSB du second operande ou second resultat
;   direct  TMP0, xxh   ; LSB du resultat intermediaire
;   direct  TMP1, xxh   ;
;   direct  TMP2, xxh   ;
;   direct  TMP3, xxh   ; MSB du resultat intermediaire

; Le chargement ou sauvegarde des resultats ou des operandes s'effectue
;   par chargement  :
;       dans la memoire externe   (ex : I32ReadExtREx),
;       dans la memoire programme (ex : I32ReadProgREx),
;       dans la memoire interne   (ex : I32ReadIntREx).
; Les octets de poid faible sont stockes en tete (avec l'adresse la plus basse).

; Les operations s'effectuent en signe (ex I32Mul) ou en non signe
;   (ex I32UMul).

; La carry (C) est positionnee en cas de debordement ou d'erreur (division)
;   pour tous les sous-programmes sauf ceux de chargement ou de sauvegarde.

; Liste des sous-programmes :   Ecrit :     Teste :
;   I32ReadExtRE1               Lecture de RE1 depuis la memoire externe
;   I32ReadExtRE2               Lecture de RE2 depuis la memoire externe
;   I32ReadProgRE1              Lecture de RE1 depuis la memoire programme
;   I32ReadProgRE2              Lecture de RE2 depuis la memoire programme
;   I32ReadIntRE1               Lecture de RE1 depuis la memoire interne
;   I32ReadIntRE2               Lecture de RE2 depuis la memoire interne
;   I32WriteExtRE1              Ecriture de RE1 dans la memoire externe
;   I32WriteExtRE2              Ecriture de RE2 dans la memoire externe
;   I32WriteIntRE1              Ecriture de RE1 dans la memoire interne
;   I32WriteIntRE2              Ecriture de RE2 dans la memoire interne
;   I32ClearRE1                 Mise a 0 de RE1
;   I32UAdd                     RE1=RE1+RE2 (non signe)
;   I32USub                     RE1=RE1-RE2 (non signe)
;   I32UMul                     RE1=RE1*RE2 (non signe)
;   I32UDiv                     RE1=RE1/RE2 et RE2=RE1%RE2 (non signe)
;   I32ShLeRE1                  RE1=RE1<<A (decalage a gauche non signe)
;   I32ShLeRE2                  RE2=RE2<<A (decalage a gauche non signe)
;   I32ShRiRE1                  RE1=RE1>>A (decalage a droite non signe)
;   I32ShRiRE2                  RE2=RE2>>A (decalage a droite non signe)
;   I32NegRE1                   RE1=-RE1
;   I32NegRE2                   RE2=-RE2
;   I32SgnRE1                   C=Signe de RE1
;   I32AbsRE1                   RE1=Valeur absolue de RE1
;   I32Add                      RE1=RE1+RE2 (signe)
;   I32Sub                      RE1=RE1-RE2 (signe)
;   I32Mul                      RE1=RE1*RE2 (signe)
;   I32Div                      RE1=RE1/RE2 et RE2=RE1%RE2 (signe)
;   I32Cvti32toi08RE1           RE1 converti de 32 a 8 bits en signe
;   I32Cvti08toi32RE1           RE1 converti de 8 a 32 bits en signe
;   I32Cvti32toi16RE1           RE1 converti de 32 a 16 bits en signe
;   I32Cvti16toi32RE1           RE1 converti de 16 a 32 bits en signe
;   I32UCvti32toi08RE1          RE1 converti de 32 a 8 bits en non signe
;   I32UCvti08toi32RE1          RE1 converti de 8 a 32 bits en non signe
;   I32UCvti32toi16RE1          RE1 converti de 32 a 16 bits en non signe
;   I32UCvti16toi32RE1          RE1 converti de 16 a 32 bits en non signe

I32ReadExtRE1:
;
; Titre     :   Lecture de 32 bits de la memoire externe vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE10,A
inc     DPTR
movx    A,@DPTR
mov     RE11,A
inc     DPTR
movx    A,@DPTR
mov     RE12,A
inc     DPTR
movx    A,@DPTR
mov     RE13,A
inc     DPTR
ret

I32ReadExtRE2:
;
; Titre     :   Lecture de 32 bits de la memoire externe vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
movx    A,@DPTR
mov     RE20,A
inc     DPTR
movx    A,@DPTR
mov     RE21,A
inc     DPTR
movx    A,@DPTR
mov     RE22,A
inc     DPTR
movx    A,@DPTR
mov     RE23,A
inc     DPTR
ret

I32ReadProgRE1:
;
; Titre     :   Lecture de 32 bits de la memoire programme vers l'operande/resultat RE1
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE10,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE11,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE12,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE13,A
inc     DPTR
ret

I32ReadProgRE2:
;
; Titre     :   Lecture de 32 bits de la memoire programme vers l'operande/resultat RE2
;               -----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@DPTR+3,@DPTR+2,@DPTR+1,@DPTR} (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     A
movc    A,@A+DPTR
mov     RE20,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE21,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE22,A
inc     DPTR
clr     A
movc    A,@A+DPTR
mov     RE23,A
inc     DPTR
ret

I32ReadIntRE1:
;
; Titre     :   Lecture de 32 bits de la memoire interne vers l'operande/resultat RE1
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE1={@R0+3,@R0+2,@R0+1,@R0} (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,@R0
inc     R0
mov     RE11,@R0
inc     R0
mov     RE12,@R0
inc     R0
mov     RE13,@R0
inc     R0
ret

I32ReadIntRE2:
;
; Titre     :   Lecture de 32 bits de la memoire interne vers l'operande/resultat RE2
;               ---------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   RE2={@R0+3,@R0+2,@R0+1,@R0} (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE20,@R0
inc     R0
mov     RE21,@R0
inc     R0
mov     RE22,@R0
inc     R0
mov     RE23,@R0
inc     R0
ret

I32WriteExtRE1:
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE1 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+3,@DPTR+2,@DPTR+1,@DPTR}=RE1 (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
movx    @DPTR,A
inc     DPTR
mov     A,RE11
movx    @DPTR,A
inc     DPTR
mov     A,RE12
movx    @DPTR,A
inc     DPTR
mov     A,RE13
movx    @DPTR,A
inc     DPTR
ret

I32WriteExtRE2:
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE2 vers la memoire externe
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans {DPH,DPL} l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@DPTR+3,@DPTR+2,@DPTR+1,@DPTR}=RE2 (DPTR incremente de 4 cases)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
movx    @DPTR,A
inc     DPTR
mov     A,RE21
movx    @DPTR,A
inc     DPTR
mov     A,RE22
movx    @DPTR,A
inc     DPTR
mov     A,RE23
movx    @DPTR,A
inc     DPTR
ret

I32WriteIntRE1:
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE1 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+3,@R0+2,@R0+1,@R0}=RE1 (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE10
inc     R0
mov     @R0,RE11
inc     R0
mov     @R0,RE12
inc     R0
mov     @R0,RE13
inc     R0
ret

I32WriteIntRE2:
;
; Titre     :   Ecriture de 32 bits de l'operande/resultat RE2 vers la memoire interne
;               ----------------------------------------------------------------------
;
; Entree(s) :   Dans R0 l'adresse de la donnee (LSB en tete)
; Sortie(s) :   {@R0+3,@R0+2,@R0+1,@R0}=RE2 (R0 incremente de 4 cases)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     @R0,RE20
inc     R0
mov     @R0,RE21
inc     R0
mov     @R0,RE22
inc     R0
mov     @R0,RE23
inc     R0
ret

I32ClearRE1:
;
; Titre     :   RAZ de l'operande/resultat RE1
;               ------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   RE1={0,0,0,0} (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE10,#0
mov     RE11,#0
mov     RE12,#0
mov     RE13,#0
clr     C
ret

I32UAdd:
;
; Titre     :   Addition non signee de deux nombres de 32 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>ffffffffh), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
mov     A,RE11
addc    A,RE21
mov     RE11,A
mov     A,RE12
addc    A,RE22
mov     RE12,A
mov     A,RE13
addc    A,RE23
mov     RE13,A
ret

I32USub:
;
; Titre     :   Soustraction non signee de deux nombres de 32 bits
;               --------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1<RE2), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
mov     A,RE11
subb    A,RE21
mov     RE11,A
mov     A,RE12
subb    A,RE22
mov     RE12,A
mov     A,RE13
subb    A,RE23
mov     RE13,A
ret

I32UMul:
;
; Titre     :   Multiplication non signee de deux nombres de 32 bits
;               ----------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2 inchange)
;               C=1 si debordement (RE1*RE2>ffffffffh), C=0 sinon
; Utilise   :   Registres TMP3 a TMP0, A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
mov     TMP0,A
mov     TMP1,RE11
mov     TMP2,RE12
mov     TMP3,RE13               ; RE1 sauve dans TMP
mov     B,RE20
mul     AB
mov     RE10,A
mov     RE11,B                  ; 1er produit stocke
mov     A,TMP1
mov     B,RE21
mul     AB
mov     RE12,A
mov     RE13,B                  ; 4ieme produit stocke
mov     A,TMP0
mov     B,RE21
mul     AB
add     A,RE11
mov     RE11,A
mov     A,B
addc    A,RE12
mov     RE12,A
mov     A,#0
addc    A,RE13
mov     RE13,A
mov     F0,C                    ; 2nd produit additionne
mov     A,TMP1
mov     B,RE20
mul     AB
add     A,RE11
mov     RE11,A
mov     A,B
addc    A,RE12
mov     RE12,A
mov     A,#0
addc    A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C                    ; 3ieme produit additionne
mov     A,TMP0
mov     B,RE22
mul     AB
add     A,RE12
mov     RE12,A
mov     A,B
addc    A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C                    ; 4ieme produit additionne
mov     A,TMP2
mov     B,RE20
mul     AB
add     A,RE12
mov     RE12,A
mov     A,B
addc    A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C                    ; 6ieme produit additionne
mov     A,TMP0
mov     B,RE23
mul     AB
add     A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C
mov     A,B
jz      I32UMul1
setb    F0
I32UMul1:                       ; 7ieme produit additionne
mov     A,TMP1
mov     B,RE22
mul     AB
add     A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C
mov     A,B
jz      I32UMul2
setb    F0
I32UMul2:                       ; 8ieme produit additionne
mov     A,TMP2
mov     B,RE21
mul     AB
add     A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C
mov     A,B
jz      I32UMul3
setb    F0
I32UMul3:                       ; 9ieme produit additionne
mov     A,TMP3
mov     B,RE20
mul     AB
add     A,RE13
mov     RE13,A
orl     C,F0
mov     F0,C
mov     A,B
jz      I32UMul4
setb    F0
I32UMul4:                       ; 10ieme produit additionne
mov     A,TMP1
jz      I32UMul5
mov     A,RE23
jz      I32UMul5
setb    F0
I32UMul5:                       ; Verification 11ieme produit nul
mov     A,TMP2
jz      I32UMul6
mov     A,RE22
jz      I32UMul6
setb    F0
I32UMul6:                       ; Verification 12ieme produit nul
mov     A,TMP3
jz      I32UMul7
mov     A,RE21
jz      I32UMul7
setb    F0
I32UMul7:                       ; Verification 13ieme produit nul
mov     A,TMP2
jz      I32UMul8
mov     A,RE23
jz      I32UMul8
setb    F0
I32UMul8:                       ; Verification 14ieme produit nul
mov     A,TMP3
jz      I32UMul9
mov     A,RE22
jz      I32UMul9
setb    F0
I32UMul9:                       ; Verification 15ieme produit nul
mov     A,TMP3
jz      I32UMulA
mov     A,RE23
jz      I32UMulA
setb    F0
I32UMulA:                       ; Verification 16ieme produit nul
mov     C,F0
ret

I32UDiv:
;
; Titre     :   Division non signee de deux nombres de 32 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur (denominateur=0 et dans ce cas RE1=ffffffffh), C=0 sinon
; Utilise   :   Registres TMP3 a TMP0, A, PSW et B non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE23
jnz     I32UDiv1                ; RE23 nul ?
mov     A,RE22
jnz     I32UDiv1                ; RE22 nul ?
mov     A,RE21
jnz     I32UDiv1                ; RE21 nul ?
mov     A,RE20
jnz     I32UDiv1                ; RE20 nul ?
mov     RE13,#ffh               ; Cas RE2=0 donc erreur
mov     RE12,#ffh
mov     RE11,#ffh
mov     RE10,#ffh
mov     RE23,#0
mov     RE22,#0
mov     RE21,#0
mov     RE20,#0
setb    C
ret
I32UDiv1:                       ; Cas normal
clr     A
xch     A,RE20
mov     TMP0,A
clr     A
xch     A,RE21
mov     TMP1,A
clr     A
xch     A,RE22
mov     TMP2,A
clr     A
xch     A,RE23
mov     TMP3,A                  ; TMP=Denominateur, RE1=Numerateur et RE2=0
mov     B,#32
I32UDiv1B:                      ; Boucle de calcul
mov     A,RE10
add     A,RE10
mov     RE10,A
mov     A,RE11
rlc     A
mov     RE11,A
mov     A,RE12
rlc     A
mov     RE12,A
mov     A,RE13
rlc     A
mov     RE13,A                  ; RE1=2*RE1
mov     A,RE20
rlc     A
mov     RE20,A
mov     A,RE21
rlc     A
mov     RE21,A
mov     A,RE22
rlc     A
mov     RE22,A
mov     A,RE23
rlc     A
mov     RE23,A                  ; RE2=2*RE2
mov     A,RE20
subb    A,TMP0
mov     A,RE21
subb    A,TMP1
mov     A,RE22
subb    A,TMP2
mov     A,RE23
subb    A,TMP3
jc      I32UDiv1S               ; RE2<TMP ?
mov     RE23,A                  ; RE2>=TMP
mov     A,RE20
subb    A,TMP0
mov     RE20,A
mov     A,RE21
subb    A,TMP1
mov     RE21,A
mov     A,RE22
subb    A,TMP2
mov     RE22,A                  ; RE2=RE2-TMP
inc     RE10
I32UDiv1S:
djnz    B,I32UDiv1B
clr     C
ret

I32ShLeRE1:
;
; Titre     :   Decalage a gauche de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#32,I32ShLeRE1T1
I32ShLeRE1T1:
jc      I32ShLeRE1T2
clr     A                       ; Decalage >=32 donc clear
mov     RE13,A
mov     RE12,A
mov     RE11,A
mov     RE10,A
ret
I32ShLeRE1T2:                   ; Decalage<32
mov     B,A
jnb     B.4,I32ShLeRE1T3
mov     RE13,RE11               ; Decalage>=16 donc shift 16 bits en bloc
mov     RE12,RE10
clr     A
mov     RE11,A
mov     RE10,A
clr     B.4
I32ShLeRE1T3:
jnb     B.3,I32ShLeRE1B
mov     RE13,RE12               ; Decalage>=8 donc shift 8 bits en bloc
mov     RE12,RE11
mov     RE11,RE10
clr     A
mov     RE10,A
clr     B.3
I32ShLeRE1B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I32ShLeRE1S
clr     C
ret
I32ShLeRE1S:
dec     B
clr     C
mov     A,RE10
rlc     A
mov     RE10,A
mov     A,RE11
rlc     A
mov     RE11,A
mov     A,RE12
rlc     A
mov     RE12,A
mov     A,RE13
rlc     A
mov     RE13,A
ljmp    I32ShLeRE1B

I32ShLeRE2:
; Titre     :   Decalage a gauche de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2<<A (decalage a gauche non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#32,I32ShLeRE2T1
I32ShLeRE2T1:
jc      I32ShLeRE2T2
clr     A                       ; Decalage >=32 donc clear
mov     RE23,A
mov     RE22,A
mov     RE21,A
mov     RE20,A
ret
I32ShLeRE2T2:                   ; Decalage<32
mov     B,A
jnb     B.4,I32ShLeRE2T3
mov     RE23,RE21               ; Decalage>=16 donc shift 16 bits en bloc
mov     RE22,RE20
clr     A
mov     RE21,A
mov     RE20,A
clr     B.4
I32ShLeRE2T3:
jnb     B.3,I32ShLeRE2B
mov     RE23,RE22               ; Decalage>=8 donc shift 8 bits en bloc
mov     RE22,RE21
mov     RE21,RE20
clr     A
mov     RE20,A
clr     B.3
I32ShLeRE2B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I32ShLeRE2S
clr     C
ret
I32ShLeRE2S:
dec     B
clr     C
mov     A,RE20
rlc     A
mov     RE20,A
mov     A,RE21
rlc     A
mov     RE21,A
mov     A,RE22
rlc     A
mov     RE22,A
mov     A,RE23
rlc     A
mov     RE23,A
ljmp    I32ShLeRE2B

I32ShRiRE1:
;
; Titre     :   Decalage a droite de l'operande/resultat RE1
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE1, nombre de decalages dans A
; Sortie(s) :   RE1=RE1>>A (decalage a droite non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#32,I32ShRiRE1T1
I32ShRiRE1T1:
jc      I32ShRiRE1T2
clr     A                       ; Decalage >=32 donc clear
mov     RE13,A
mov     RE12,A
mov     RE11,A
mov     RE10,A
ret
I32ShRiRE1T2:                   ; Decalage<32
mov     B,A
jnb     B.4,I32ShRiRE1T3
mov     RE11,RE13               ; Decalage>=16 donc shift 16 bits en bloc
mov     RE10,RE12
clr     A
mov     RE13,A
mov     RE12,A
clr     B.4
I32ShRiRE1T3:
jnb     B.3,I32ShRiRE1B
mov     RE10,RE11               ; Decalage>=8 donc shift 8 bits en bloc
mov     RE11,RE12
mov     RE12,RE13
clr     A
mov     RE13,A
clr     B.3
I32ShRiRE1B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I32ShRiRE1S
clr     C
ret
I32ShRiRE1S:
dec     B
clr     C
mov     A,RE13
rrc     A
mov     RE13,A
mov     A,RE12
rrc     A
mov     RE12,A
mov     A,RE11
rrc     A
mov     RE11,A
mov     A,RE10
rrc     A
mov     RE10,A
ljmp    I32ShRiRE1B

I32ShRiRE2:
;
; Titre     :   Decalage a droite de l'operande/resultat RE2
;               --------------------------------------------
;
; Entree(s) :   Donnee dans RE2, nombre de decalages dans A
; Sortie(s) :   RE2=RE2>>A (decalage a droite non signe)
;               Avec C=0 meme si debordement, A perdu
; Utilise   :   Registres A,B et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
cjne    A,#32,I32ShRiRE2T1
I32ShRiRE2T1:
jc      I32ShRiRE2T2
clr     A                       ; Decalage >=32 donc clear
mov     RE23,A
mov     RE22,A
mov     RE21,A
mov     RE20,A
ret
I32ShRiRE2T2:                   ; Decalage<32
mov     B,A
jnb     B.4,I32ShRiRE2T3
mov     RE21,RE23               ; Decalage>=16 donc shift 16 bits en bloc
mov     RE20,RE22
clr     A
mov     RE23,A
mov     RE22,A
clr     B.4
I32ShRiRE2T3:
jnb     B.3,I32ShRiRE2B
mov     RE20,RE21               ; Decalage>=8 donc shift 8 bits en bloc
mov     RE21,RE22
mov     RE22,RE23
clr     A
mov     RE23,A
clr     B.3
I32ShRiRE2B:                    ; Boucle pour les 0 a 7 derniers decalages
mov     A,B
jnz     I32ShRiRE2S
clr     C
ret
I32ShRiRE2S:
dec     B
clr     C
mov     A,RE23
rrc     A
mov     RE23,A
mov     A,RE22
rrc     A
mov     RE22,A
mov     A,RE21
rrc     A
mov     RE21,A
mov     A,RE20
rrc     A
mov     RE20,A
ljmp    I32ShRiRE2B

I32NegRE1:
;
; Titre     :   Changement de signe de l'operande/resultat RE1
;               ----------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=-RE1
;               Avec C=1 si debordement (si RE1=80000000h=-2147483648) et C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A
clr     C
xrl     A,#80h
jnz     I32NegRE1F              ; Test si RE13=80h
mov     A,RE12
jnz     I32NegRE1F              ; Test si RE12=00h
mov     A,RE11
jnz     I32NegRE1F              ; Test si RE11=00h
mov     A,RE10
jnz     I32NegRE1F              ; Test si RE10=00h
setb    C                       ; C=1 car RE1=80000000h (cas de debordement)
I32NegRE1F:
ret

I32NegRE2:
;
; Titre     :   Changement de signe de l'operande/resultat RE2
;               ----------------------------------------------
;
; Entree(s) :   Donnee dans RE2
; Sortie(s) :   RE2=-RE2
;               Avec C=1 si debordement (si RE2=80000000h=-2147483648) et C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
mov     A,RE22
cpl     A
addc    A,#0
mov     RE22,A
mov     A,RE23
cpl     A
addc    A,#0
mov     RE23,A
clr     C
xrl     A,#80h
jnz     I32NegRE2F              ; Test si RE23=80h
mov     A,RE22
jnz     I32NegRE2F              ; Test si RE22=00h
mov     A,RE21
jnz     I32NegRE2F              ; Test si RE21=00h
mov     A,RE20
jnz     I32NegRE2F              ; Test si RE20=00h
setb    C                       ; C=1 car RE2=80000000h (cas de debordement)
I32NegRE2F:
ret

I32SgnRE1:
;
; Titre     :   Recuperation du signe de l'operande/resultat RE1
;               ------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   C=0 si le nombre est positif
;               C=1 si le nombre est negatif
;               RE1 reste inchange
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
mov     C,ACC.7
ret

I32AbsRE1:
;
; Titre     :   Valeur absolue de l'operande/resultat RE1
;               -----------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   RE1=ABS(RE1)
;               C=1 si debordement en codage signe (RE1=80000000=-2147483648)
;               C=0 si nombre compatible d'un codage signe
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   I32SgnRE1
jnc     I32AbsRE1F
ljmp    I32NegRE1               ; Change le signe du nombre negatif
I32AbsRE1F:                     ; Rien a faire car nombre positif
ret

I32Add:
;
; Titre     :   Addition signee de deux nombres de 32 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1+RE2 (RE2 inchange)
;               C=1 si debordement (RE1+RE2>7fffffffh ou <80000000h), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
add     A,RE20
mov     RE10,A
mov     A,RE11
addc    A,RE21
mov     RE11,A
mov     A,RE12
addc    A,RE22
mov     RE12,A
mov     A,RE13
addc    A,RE23
mov     RE13,A
mov     C,OV
ret

I32Sub:
;
; Titre     :   Soustraction signee de deux nombres de 32 bits
;               ----------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1-RE2 (RE2 inchange)
;               C=1 si debordement (RE1-RE2>7fffffffh ou <80000000h), C=0 sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
clr     C
mov     A,RE10
subb    A,RE20
mov     RE10,A
mov     A,RE11
subb    A,RE21
mov     RE11,A
mov     A,RE12
subb    A,RE22
mov     RE12,A
mov     A,RE13
subb    A,RE23
mov     RE13,A
mov     C,OV
ret

I32Mul:
;
; Titre     :   Multiplication signee de deux nombres de 32 bits
;               ------------------------------------------------
;
; Entree(s) :   Donnees dans RE1 et RE2
; Sortie(s) :   RE1=RE1*RE2 (RE2=abs(RE2))
;               C=1 si debordement (RE1*RE2>7fffffffh ou <80000000h), C=0 sinon
; Utilise   :   Registres TMP3, TMP2, TMP1, TMP0, A, PSW et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
jb      ACC.7,I32MulS
mov     A,RE23                  ; RE1 positif
jb      ACC.7,I32MulNonPositif
ljmp    I32MulPositif
I32MulNonPositif:
mov     A,RE20                  ; RE20 negatif donc change de signe
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
mov     A,RE22
cpl     A
addc    A,#0
mov     RE22,A
mov     A,RE23
cpl     A
addc    A,#0
mov     RE23,A
I32MulNegatif:                  ; Le resultat est negatif
lcall   I32UMul
jc      I32MulE2                ; Verification pas de depassement de 4294967296
mov     A,RE10
jnz     I32MulNN
mov     A,RE11
jnz     I32MulNN
mov     A,RE12
jnz     I32MulNN
mov     A,RE13
jnz     I32MulNN
ret                             ; Cas d'un resultat nul (0*(-qq chose))
I32MulNN:                       ; Resultat non nul
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A                  ; Change le signe du resultat
jnb     ACC.7,I32MulE           ; Verification pas de depassement de -2147483648
clr     C
ret
I32MulE2:                       ; Erreur avant le changement de signe
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A                  ; Change le signe du resultat
I32MulE:                        ; Erreur
setb    C
ret
I32MulS:                        ; RE10 negatif donc change de signe
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A
mov     A,RE23
jnb     ACC.7,I32MulNegatif
mov     A,RE20                  ; RE20 negatif donc change de signe
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
mov     A,RE22
cpl     A
addc    A,#0
mov     RE22,A
mov     A,RE23
cpl     A
addc    A,#0
mov     RE23,A
I32MulPositif:                  ; Le resultat est positif
lcall   I32UMul
jc      I32MulFin               ; Verification pas de depassement de 4294967296
mov     A,RE13
mov     C,ACC.7                 ; Donc C=1 si Resultat>=2147483648
I32MulFin:
ret

I32Div:
;
; Titre     :   Division signee de deux nombres de 32 bits
;               ------------------------------------------
;
; Entree(s) :   Donnees dans RE1 (numerateur) et RE2 (denominateur)
; Sortie(s) :   RE1=RE1/RE2 et RE2=RE1%RE2
;               C=1 si erreur : - denominateur=0 et dans ce cas RE1=7fffffffh ou 80000000h
;                                 selon le signe de RE1)
;                               - debordement du quotien en signe (+214748348)
;               C=0 si pas d'erreur
; Utilise   :   Registres TMP3, TMP2, TMP1, TMP0, A, PSW et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,RE20
jnz     I32Div1                 ; RE20 nul ?
mov     A,RE21
jnz     I32Div1                 ; RE21 nul ?
mov     A,RE22
jnz     I32Div1                 ; RE22 nul ?
mov     A,RE23
jnz     I32Div1                 ; RE23 nul ?
mov     RE20,#0
mov     RE21,#0
mov     RE22,#0
mov     RE23,#0
mov     A,RE13
jb      ACC.7,I32DivE1
mov     RE13,#7fh               ; Cas RE2=0 et RE1>=0
mov     RE12,#ffh
mov     RE11,#ffh
mov     RE10,#ffh
setb    C
ret
I32DivE1:                       ; Cas RE2=0 et RE1<0
mov     RE13,#80h
mov     RE12,#00h
mov     RE11,#00h
mov     RE10,#00h
setb    C
ret
I32Div1:                        ; Cas normal RE2!=0
mov     A,RE23
jb      ACC.7,I32DivS1
mov     A,RE13                  ; RE2>0
jb      ACC.7,I32DivS2
ljmp    I32UDiv1                ; RE2>0 et RE1>=0
I32DivS2:                       ; RE2>0 et RE1<0
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A                  ; Changement de signe de RE1
I32DivS3:
lcall   I32UDiv1
mov     A,RE10
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A                  ; Changement de signe de RE1
mov     A,RE20
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
mov     A,RE22
cpl     A
addc    A,#0
mov     RE22,A
mov     A,RE23
cpl     A
addc    A,#0
mov     RE23,A                  ; Changement de signe du reste
clr     C
ret
I32DivS1:                       ; RE2<0
mov     A,RE20
cpl     A
add     A,#1
mov     RE20,A
mov     A,RE21
cpl     A
addc    A,#0
mov     RE21,A
mov     A,RE22
cpl     A
addc    A,#0
mov     RE22,A
mov     A,RE23
cpl     A
addc    A,#0
mov     RE23,A                  ; Changement de signe du reste
mov     A,RE13
jnb     ACC.7,I32DivS3          ; RE1>=0
mov     A,RE10                  ; RE2<0 et RE1<0
cpl     A
add     A,#1
mov     RE10,A
mov     A,RE11
cpl     A
addc    A,#0
mov     RE11,A
mov     A,RE12
cpl     A
addc    A,#0
mov     RE12,A
mov     A,RE13
cpl     A
addc    A,#0
mov     RE13,A                  ; Changement de signe de RE1
lcall   I32UDiv1
mov     A,RE13
jnb     ACC.7,I32DivS4
setb    C                       ; Erreur de debordement : (-214748348)/(-1)=(+214748348)
I32DivS4:
ret

I32Cvti32toi08RE1:
;
; Titre     :   Conversion signee de 32 bits vers 8 bits dans RE1
;               -------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 8 bits dans RE10
;               Avec C=1 si RE1>127 ou RE1<-128, C=0 sinon
;               Dans tous les cas RE10=RE1[256] et RE11=0 si >=0 RE11=ffh sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
jb      ACC.7,I32Cvti32toi08RE1S
mov     A,RE10                  ; Nombre positif
mov     C,ACC.7                 ; Donc C=1 si > 127[256]
mov     A,RE11
jnz     I32Cvti32toi08RE1F
mov     A,RE12
jnz     I32Cvti32toi08RE1F
mov     A,RE13
jnz     I32Cvti32toi08RE1F
mov     RE13,#0
mov     RE12,#0
mov     RE11,#0
ret
I32Cvti32toi08RE1F:             ; C=1 car >255
mov     RE13,#0
mov     RE12,#0
mov     RE11,#0
setb    C
ret
I32Cvti32toi08RE1S:             ; Nombre negatif
mov     A,RE10
mov     C,ACC.7
cpl     C                       ; Donc C=1 si <-128[256]
mov     A,RE13
inc     A
jnz     I32Cvti32toi08RE1G
mov     A,RE12
inc     A
jnz     I32Cvti32toi08RE1G
mov     A,RE11
inc     A
jnz     I32Cvti32toi08RE1G
mov     RE13,#ffh
mov     RE12,#ffh
mov     RE11,#ffh
ret
I32Cvti32toi08RE1G:             ; C=1 car <-256
mov     RE13,#ffh
mov     RE12,#ffh
mov     RE11,#ffh
setb    C
ret

I32Cvti08toi32RE1:
;
; Titre     :   Conversion signee de 8 bits vers 32 bits dans RE1
;               -------------------------------------------------
;
; Entree(s) :   Donnee dans RE10
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 (C=0)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE10
clr     C
jb      ACC.7,I32Cvti08toi32RE1S
mov     RE13,#0                 ; Nombre positif
mov     RE12,#0
mov     RE11,#0
ret
I32Cvti08toi32RE1S:
mov     RE13,#ffh               ; Nombre negatif
mov     RE12,#ffh
mov     RE11,#ffh
ret

I32Cvti32toi16RE1:
;
; Titre     :   Conversion signee de 32 bits vers 16 bits dans RE1
;               --------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 16 bits dans {RE11,RE10}
;               Avec C=1 si RE1>32767 ou RE1<-32768, C=0 sinon
;               Dans tous les cas {RE11,RE10}=RE1[65536] et RE13=RE12=0 si >=0 RE13=RE12=ffh sinon
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
jb      ACC.7,I32Cvti32toi16RE1S
mov     A,RE11                  ; Nombre positif
mov     C,ACC.7                 ; Donc C=1 si > 32767[65536]
mov     A,RE12
jnz     I32Cvti32toi16RE1F
mov     A,RE13
jnz     I32Cvti32toi16RE1F
mov     RE13,#0
mov     RE12,#0
ret
I32Cvti32toi16RE1F:             ; C=1 car >65535
mov     RE13,#0
mov     RE12,#0
setb    C
ret
I32Cvti32toi16RE1S:             ; Nombre negatif
mov     A,RE11
mov     C,ACC.7
cpl     C                       ; Donc C=1 si <-32768[65536]
mov     A,RE13
inc     A
jnz     I32Cvti32toi16RE1G
mov     A,RE12
inc     A
jnz     I32Cvti32toi16RE1G
mov     RE13,#ffh
mov     RE12,#ffh
ret
I32Cvti32toi16RE1G:             ; C=1 car <-65536
mov     RE13,#ffh
mov     RE12,#ffh
setb    C
ret

I32Cvti16toi32RE1:
;
; Titre     :   Conversion signee de 16 bits vers 32 bits dans RE1
;               --------------------------------------------------
;
; Entree(s) :   Donnee dans {RE11,RE10}
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 (C=0)
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE11
clr     C
jb      ACC.7,I32Cvti16toi32RE1S
mov     RE13,#0                 ; Nombre positif
mov     RE12,#0
ret
I32Cvti16toi32RE1S:
mov     RE13,#ffh               ; Nombre negatif
mov     RE12,#ffh
ret

I32UCvti32toi08RE1:
;
; Titre     :   Conversion non signee de 32 bits vers 8 bits dans RE1
;               -----------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 8 bits dans RE10
;               Avec C=1 si debordement (RE1>255), C=0 sinon
;               Dans tous les cas RE10=RE1[256] et {RE13,RE12,RE11}={0,0,0}
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
jnz     I32UCvti32toi08RE1S
mov     A,RE12
jnz     I32UCvti32toi08RE1S
mov     A,RE11
jnz     I32UCvti32toi08RE1S
mov     RE13,#0                 ; C=0 car RE13=RE12=RE11=0
mov     RE12,#0
mov     RE11,#0
clr     C
ret
I32UCvti32toi08RE1S:            ; C=1 car {RE13,RE12,RE11}!={0,0,0}
mov     RE13,#0
mov     RE12,#0
mov     RE11,#0
setb    C
ret

I32UCvti08toi32RE1:
;
; Titre     :   Conversion non signee de 8 bits vers 32 bits dans RE1
;               -----------------------------------------------------
;
; Entree(s) :   Donnee dans RE10
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE13,#0
mov     RE12,#0
mov     RE11,#0
clr     C
ret

I32UCvti32toi16RE1:
;
; Titre     :   Conversion non signee de 32 bits vers 16 bits dans RE1
;               ------------------------------------------------------
;
; Entree(s) :   Donnee dans RE1
; Sortie(s) :   Donnee convertie sur 16 bits dans {RE11,RE10}
;               Avec C=1 si debordement (RE1>65535), C=0 sinon
;               Dans tous les cas {RE11,RE10}=RE1[65536] et {RE13,RE12}={0,0}
; Utilise   :   Registres A et PSW non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,RE13
jnz     I32UCvti32toi16RE1S
mov     A,RE12
jnz     I32UCvti32toi16RE1S
mov     RE13,#0                 ; C=0 car RE13=RE12=0
mov     RE12,#0
clr     C
ret
I32UCvti32toi16RE1S:            ; C=1 car {RE13,RE12}!={0,0}
mov     RE13,#0
mov     RE12,#0
setb    C
ret

I32UCvti16toi32RE1:
;
; Titre     :   Conversion non signee de 16 bits vers 32 bits dans RE1
;               ------------------------------------------------------
;
; Entree(s) :   Donnee dans {RE11,RE10}
; Sortie(s) :   Donnee convertie sur 32 bits dans RE1 (C=0)
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     RE13,#0
mov     RE12,#0
clr     C
ret
