; LIBKBD1.ASM
;
;   Bibliotheque de sous-programmes de gestion d'un clavier PC/AT (AZERTY)
;   ----------------------------------------------------------------------
;
;                          (C) E+E CHAMOUARD 1998
;

; A definir dans le programme principal :
;   bit     KbdClk              Bit connecte au signal Clk du clavier
;                               (P3.2 ou P3.3 sur un MCS51 "de base")
;   bit     KbdDat              Bit connecte au signal Data du clavier
;   bit     KbdIT               Interuption utilisee (IT0 ou IT1)
;   bit     KbdEX               Interuption utilisee (EX0 ou EX1)
;                               (IT0 et EX0 si P3.2 et IT1 et EX1 si P3.3)
;   byte    KbdDel              Delai du time out du clavier 
;                               A ajuster en fonction de la vitesse
;                               du clavier et du processeur
;                               (10 mesure sur un 8051 a 12 MHz par ex.)

;   word    KbdBufferStart      Adresse de debut du buffer clavier (memoire externe)
;   word    KbdBufferEnd        Adresse de fin du buffer clavier (debut + ffh)

;   direct  KbdIndexRead        Index de lecture du buffer clavier
;   direct  KbdIndexWrite       Index d'ecriture du buffer clavier
;   direct  KbdReadState        Etat de la machine d'etat de reception
;   direct  KbdReadFlags        Flags de reception
;   direct  KbdReadModif        Modificateur de reception

; Pour faire fonctionner le clavier, le programme principal doit :
;   1) initialiser les interuptions clavier, par exemple :
;       lcall   KbdInit         ; Init clavier
;       setb    KbdIT           ; It sur front descendant
;       setb    KbdEX           ; It externe 0 autorisee
;       setb    EA              ; It autorisees
;   2) avoir une routine d'interuption de la forme :
;       ORG 0003h               ; Exemple avec EX0
;       clr     KbdIT           ; Arret ITx
;       push    PSW
;       push    ACC
;       push    B
;       push    DPL
;       push    DPH
;       lcall   KbdInput        ; Entree du clavier
;       lcall   KbdInputTrt     ; Conversion en ASCII
;       pop     DPH
;       pop     DPL
;       pop     B
;       pop     ACC
;       pop     PSW
;       setb    KbdIT           ; Mise en route IT
;       reti
;   3) tester la presence d'un caractere dans le buffer de reception
;       lcall   KbdKbit
;       jz      qsdfqsdf
;   4) recuperer les caracteres dans le buffer de reception
;       lcall   KbdGetch


;   Definition des constantes
;   -------------------------

byte    KbdStateIdle,       #00h
byte    KbdStateE0,         #01h
byte    KbdStateF0,         #02h
byte    KbdStateE0F0,       #03h
; Modifs faites suite pb sur pave intermediaire
;byte    KbdStateE012,       #04h
;byte    KbdStateE012E0,     #05h
; Fin des modifs faites suite pb sur pave intermediaire
byte    KbdStateE1,         #06h
byte    KbdStateE114,       #07h
byte    KbdStateE1F0,       #08h

bit     KbdFlagScrolllock,  ACC.0
bit     KbdFlagNumlock,     ACC.1
bit     KbdFlagCapslock,    ACC.2

bit     KbdModifLShift,     ACC.0
bit     KbdModifRShift,     ACC.1
bit     KbdModifLCtl,       ACC.2
bit     KbdModifRCtl,       ACC.3
bit     KbdModifAlt,        ACC.4
bit     KbdModifAltgr,      ACC.5

KbdInit:
;
; Titre     :   Initialisation de l'interface clavier
;               -------------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   RAZ du tampon de caracteres
;               RAZ de la machine d'etat
;               RAZ des LEDs du clavier
; Utilise   :   Registres A et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     KbdReadState,#KbdStateIdle
mov     KbdReadFlags,#0
mov     KbdReadModif,#0 ; Reset num lock, caps lock et scroll lock
mov     KbdIndexRead,KbdIndexWrite
mov     A,#ffh
lcall   KbdOutput       ; Reset du clavier
ret

KbdSetNumlock:
;
; Titre     :   Mise en fonction du num lock du clavier
;               ---------------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   Led du clavier et flag correspondant
; Utilise   :   Registres A et B non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
mov     A,KbdReadFlags
setb    KbdFlagNumlock
mov     KbdReadFlags,A
mov     A,#edh
lcall   KbdOutput
mov     A,KbdReadFlags
lcall   KbdOutput                   ; Affichage des LEDs
ret

KbdKbhit:
;
; Titre     :   Determine si un char attend dans le buffer clavier
;               --------------------------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   Nombre de caracteres dans la FIFO dans le registre A
; Utilise   :   Rien
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     A,KbdIndexWrite
clr     C
subb    A,KbdIndexRead
clr     C
ret

KbdGetch:
;
; Titre     :   Lecture d'un caractere dans le buffer
;               -------------------------------------
;
; Entree(s) :   Rien
;               Ne verifie pas si un caractere est dans le buffer
; Sortie(s) :   Caractere dans le registre A
; Utilise   :   Registres DPH et DPL non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     DPTR,#KbdBufferStart
mov     A,KbdIndexRead
add     A,DPL
mov     DPL,A
mov     A,#0
addc    A,DPH
mov     DPH,A
movx    A,@DPTR
inc     KbdIndexRead
ret

KbdAttAndGetch:
;
; Titre     :   Lecture d'un caractere au clavier avec attente
;               ----------------------------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   Caractere dans le registre A (attente infinie)
; Utilise   :   Registres DPH et DPL non sauvegardes
; Pile      :   4 avec l'appel
; Cycles    :   ?
;
lcall   KbdKbhit
jz      KbdAttAndGetch
ljmp    KbdGetch

KbdAttAndDrop:
;
; Titre     :   Attente frappe au clavier
;               -------------------------
;
; Entree(s) :   Rien
; Sortie(s) :   Rien (carac perdu et attente infinie)
;               Si caractere etendu, attend le second caractere
; Utilise   :   Registres A, DPH et DPL non sauvegardes
; Pile      :   6 avec l'appel
; Cycles    :   ?
;
lcall   KbdAttAndGetch
jnz     KbdAttAndDropOut
KbdAttAndDropSuite:     ; Si premier carac=0 attente d'un second
lcall   KbdAttAndGetch
KbdAttAndDropOut:
ret

KbdInput:
;
; Titre     :   Entree en provenance du clavier
;               -------------------------------
;
; Entree(s) :   Sous prog appele avec KbdDat a 0 (par ex sous it)
;               Ou attente indefinie du caractere (pour l'instant)
; Sortie(s) :   Code clavier dans le registre A
;                   00h si erreur de parite
;               A faire : time out ?
; Utilise   :   Registres B et DPH non sauvegardes
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
mov     DPH,#KbdDel      ; Attente clock a 0
KbdInputB1:
jnb     KbdClk,KbdInputS1
djnz    DPH,KbdInputB1
ljmp    KbdInputNOk
KbdInputS1:
mov     DPH,#KbdDel     ; Attente clock a 1
KbdInputB2:
jb      KbdClk,KbdInputS2
djnz    DPH,KbdInputB2
ljmp    KbdInputNOk
KbdInputS2:
mov     B,#8
mov     A,#0
KbdInputBoucle:
mov     DPH,#KbdDel     ; Attente clock a 0
KbdInputB3:
jnb     KbdClk,KbdInputS3
djnz    DPH,KbdInputB3
ljmp    KbdInputNOk
KbdInputS3:
mov     C,KbdDat        ; Lecture du bit
rrc     A               ; Stock dans accumulateur LSB first
mov     DPH,#KbdDel     ; Attente clock a 1
KbdInputB4:
jb      KbdClk,KbdInputS4
djnz    DPH,KbdInputB4
ljmp    KbdInputNOk
KbdInputS4:
djnz    B,KbdInputBoucle
mov     DPH,#KbdDel     ; Attente clock a 0
KbdInputB5:
jnb     KbdClk,KbdInputS5
djnz    DPH,KbdInputB5
ljmp    KbdInputNOk
KbdInputS5:
mov     C,KbdDat        ; Lecture du bit de parite
mov     DPH,#KbdDel     ; Attente clock a 1
KbdInputB6:
jb      KbdClk,KbdInputS6
djnz    DPH,KbdInputB6
ljmp    KbdInputNOk
KbdInputS6:
mov     DPH,#KbdDel     ; Attente clock a 0
KbdInputB7:
jnb     KbdClk,KbdInputS7
djnz    DPH,KbdInputB7
ljmp    KbdInputNOk
KbdInputS7:
                        ; Lecture du stop bit
mov     DPH,#KbdDel     ; Attente clock a 1
KbdInputB8:
jb      KbdClk,KbdInputS8
djnz    DPH,KbdInputB8
ljmp    KbdInputNOk
KbdInputS8:
jc      KbdInputPis1
jnb     P,KbdInputNOk   ; C=P=0 donc NOk
sjmp    KbdInputOk      ; C=0, P=1 donc Ok
KbdInputPis1:           ; C=1
jnb     P,KbdInputOk    ; C=1, P=0 donc Ok
KbdInputNOk:            ; C=P donc NOk
mov     A,#00h
KbdInputOk:
ret

KbdInputTrt:
;
; Titre     :   Traitement de l'entree brute du clavier
;               ---------------------------------------
;
; Entree(s) :   Entree brute du clavier dans le registre A
; Sortie(s) :   Rien
;               Mise a jour de l'index du buffer (KbdIndexWrite)
;               et/ou mise a jour de la machine d'etat et des flags
; Utilise   :   Registres DPL, DPH, B et A non sauvegardes
; Pile      :   7 avec l'appel
; Cycles    :   ?
;
; !!!!!!!!!!!!!!!! manque traitement du ^,~,` et du 
push    ACC
mov     DPTR,#KbdInputTrtJump
mov     A,KbdReadState
anl     A,#00111111b
rl      A
rl      A
jmp     @A+DPTR             ; Aiguillage en fonction de l'etat anterieur
KbdInputTrtJump:
ljmp    KbdInputTrtIdle
nop
ljmp    KbdInputTrtE0
nop
ljmp    KbdInputTrtF0
nop
ljmp    KbdInputTrtE0F0
nop
; Modifs faites suite pb sur pave intermediaire
ljmp    KbdInputTrtIdle ; KbdInputTrtE012
nop
ljmp    KbdInputTrtIdle ; KbdInputTrtE012E0
nop
; Fin des modifs faites suite pb sur pave intermediaire
ljmp    KbdInputTrtE1
nop
ljmp    KbdInputTrtE114
nop
ljmp    KbdInputTrtE1F0

KbdInputTrtIdle:                    ; Machine en etat Idle
pop     ACC                         ; --------------------
cjne    A,#F0h,KbdInputTrtIdleS1
mov     KbdReadState,#KbdStateF0    ; Passage a l'etat F0
ret
KbdInputTrtIdleS1:
cjne    A,#E0h,KbdInputTrtIdleS2
mov     KbdReadState,#KbdStateE0    ; Passage a l'etat E0
ret
KbdInputTrtIdleS2:
cjne    A,#E1h,KbdInputTrtIdleS3
mov     KbdReadState,#KbdStateE1    ; Passage a l'etat E1
ret
KbdInputTrtIdleS3:
cjne    A,#14h,KbdInputTrtIdleS4
mov     KbdReadState,#KbdStateIdle  ; Touche LCtl
mov     A,KbdReadModif
setb    KbdModifLCtl
mov     KbdReadModif,A
ret
KbdInputTrtIdleS4:
cjne    A,#11h,KbdInputTrtIdleS5
mov     KbdReadState,#KbdStateIdle  ; Touche Alt
mov     A,KbdReadModif
setb    KbdModifAlt
mov     KbdReadModif,A
ret
KbdInputTrtIdleS5:
cjne    A,#12h,KbdInputTrtIdleS6
mov     KbdReadState,#KbdStateIdle  ; Touche LShift
mov     A,KbdReadModif
setb    KbdModifLShift
mov     KbdReadModif,A
ret
KbdInputTrtIdleS6:
cjne    A,#59h,KbdInputTrtIdleS7
mov     KbdReadState,#KbdStateIdle  ; Touche RShift
mov     A,KbdReadModif
setb    KbdModifRShift
mov     KbdReadModif,A
ret
KbdInputTrtIdleS7:
cjne    A,#58h,KbdInputTrtIdleS8
mov     KbdReadState,#KbdStateIdle  ; Touche CapsLock
mov     A,KbdReadFlags
cpl     KbdFlagCapslock
mov     KbdReadFlags,A
mov     A,#edh
lcall   KbdOutput
mov     A,KbdReadFlags
lcall   KbdOutput                   ; Affichage des LEDs
ret
KbdInputTrtIdleS8:
cjne    A,#77h,KbdInputTrtIdleS9
mov     KbdReadState,#KbdStateIdle  ; Touche NumLock
mov     A,KbdReadFlags
cpl     KbdFlagNumlock
mov     KbdReadFlags,A
mov     A,#edh
lcall   KbdOutput
mov     A,KbdReadFlags
lcall   KbdOutput                   ; Affichage des LEDs
ret
KbdInputTrtIdleS9:                  ; Traitement touches normales
;lcall   KbdPutch                    ; Test sortie directe
;ret                                 ; Test sortie directe
push    ACC
mov     DPTR,#KbdInputMap
mov     B,#8
mul     AB
add     A,DPL
mov     DPL,A
mov     A,B
addc    A,DPH
mov     DPH,A                       ; DPTR pointe sur la touche
pop     ACC
cjne    A,#84h,KbdInputTrtIdleS14
KbdInputTrtIdleS14:
jnc     KbdInputTrtIdleFin          ; Si > 83h sortie
cjne    A,#69h,KbdInputTrtIdleS10
KbdInputTrtIdleS10:
jc      KbdInputTrtIdleNonNum       ; Si < 69h pas pave num
cjne    A,#7eh,KbdInputTrtIdleS11
KbdInputTrtIdleS11:
jc      KbdInputTrtIdleNum          ; Si >= 69 h et < 7eh pave num
KbdInputTrtIdleNonNum:              ; Cas non pave num
mov     A,KbdReadModif
jb      KbdModifAlt,KbdInputTrtAlt
jb      KbdModifAltgr,KbdInputTrtAlt
jb      KbdModifLCtl,KbdInputTrtCtl
jb      KbdModifRCtl,KbdInputTrtCtl
jb      KbdModifLShift,KbdInputTrtShift
jb      KbdModifRShift,KbdInputTrtShift
mov     A,KbdReadFlags
jb      KbdFlagCapslock,KbdInputTrtShift
sjmp    KbdInputTrtNorm
KbdInputTrtIdleNum:                 ; Cas pave num
mov     A,KbdReadModif
jb      KbdModifAlt,KbdInputTrtAlt
jb      KbdModifAltgr,KbdInputTrtAlt
jb      KbdModifLCtl,KbdInputTrtCtl
jb      KbdModifRCtl,KbdInputTrtCtl
jb      KbdModifLShift,KbdInputTrtShift
jb      KbdModifRShift,KbdInputTrtShift
mov     A,KbdReadFlags
jb      KbdFlagNumlock,KbdInputTrtShift
sjmp    KbdInputTrtNorm
KbdInputTrtAlt:
inc     DPTR
inc     DPTR
KbdInputTrtCtl:
inc     DPTR
inc     DPTR
KbdInputTrtShift:
inc     DPTR
inc     DPTR
KbdInputTrtNorm:
mov     A,#0
movc    A,@A+DPTR
cjne    A,#ffh,KbdInputTrtIdleS12
sjmp    KbdInputTrtIdleFin          ; Si ff sortie
KbdInputTrtIdleS12:
push    DPH
push    DPL
lcall   KbdPutch
pop     DPL
pop     DPH
jnz     KbdInputTrtIdleFin          ; Sortie d'un seul byte
inc     DPTR                        ; Un 2nd byte a sortir
mov     A,#0
movc    A,@A+DPTR
cjne    A,#ffh,KbdInputTrtIdleS13
sjmp    KbdInputTrtIdleFin          ; Si ff sortie
KbdInputTrtIdleS13:
lcall   KbdPutch
KbdInputTrtIdleFin:
ret

KbdInputTrtE0:                      ; Machine en etat E0
pop     ACC                         ; ------------------
cjne    A,#F0h,KbdInputTrtE0S1
mov     KbdReadState,#KbdStateE0F0  ; Passage a l'etat E0F0
ret
KbdInputTrtE0S1:
cjne    A,#12h,KbdInputTrtE0S2
; Modifs faites suite pb sur pave intermediaire
mov     KbdReadState,#KbdStateIdle  ; Passage etat idle (en fait E012)
;mov     KbdReadState,#KbdStateE012  ; Passage etat E012
; Fin des mmodifs faites suite pb sur pave intermediaire
ret
KbdInputTrtE0S2:
cjne    A,#11h,KbdInputTrtE0S3
mov     KbdReadState,#KbdStateIdle  ; Touche AltGr
mov     A,KbdReadModif
setb    KbdModifAltgr
mov     KbdReadModif,A
ret
KbdInputTrtE0S3:
cjne    A,#14h,KbdInputTrtE0S4
mov     KbdReadState,#KbdStateIdle  ; Touche RCtl
mov     A,KbdReadModif
setb    KbdModifRCtl
mov     KbdReadModif,A
ret
KbdInputTrtE0S4:
cjne    A,#70h,KbdInputTrtE0S5
mov     KbdReadState,#KbdStateIdle  ; Touche Insert
mov     A,#0
lcall   KbdPutch
mov     A,#52h
lcall   KbdPutch
ret
KbdInputTrtE0S5:
cjne    A,#71h,KbdInputTrtE0S6
mov     KbdReadState,#KbdStateIdle  ; Touche Delete
mov     A,#0
lcall   KbdPutch
mov     A,#53h
lcall   KbdPutch
ret
KbdInputTrtE0S6:
cjne    A,#6bh,KbdInputTrtE0S7
mov     KbdReadState,#KbdStateIdle  ; Touche Left
mov     A,#0
lcall   KbdPutch
mov     A,#4bh
lcall   KbdPutch
ret
KbdInputTrtE0S7:
cjne    A,#6ch,KbdInputTrtE0S8
mov     KbdReadState,#KbdStateIdle  ; Touche Home
mov     A,#0
lcall   KbdPutch
mov     A,#47h
lcall   KbdPutch
ret
KbdInputTrtE0S8:
cjne    A,#69h,KbdInputTrtE0S9
mov     KbdReadState,#KbdStateIdle  ; Touche End
mov     A,#0
lcall   KbdPutch
mov     A,#4fh
lcall   KbdPutch
ret
KbdInputTrtE0S9:
cjne    A,#75h,KbdInputTrtE0S10
mov     KbdReadState,#KbdStateIdle  ; Touche Up
mov     A,#0
lcall   KbdPutch
mov     A,#48h
lcall   KbdPutch
ret
KbdInputTrtE0S10:
cjne    A,#72h,KbdInputTrtE0S11
mov     KbdReadState,#KbdStateIdle  ; Touche Down
mov     A,#0
lcall   KbdPutch
mov     A,#50h
lcall   KbdPutch
ret
KbdInputTrtE0S11:
cjne    A,#7dh,KbdInputTrtE0S12
mov     KbdReadState,#KbdStateIdle  ; Touche Page Up
mov     A,#0
lcall   KbdPutch
mov     A,#49h
lcall   KbdPutch
ret
KbdInputTrtE0S12:
cjne    A,#7ah,KbdInputTrtE0S13
mov     KbdReadState,#KbdStateIdle  ; Touche Page Down
mov     A,#0
lcall   KbdPutch
mov     A,#51h
lcall   KbdPutch
ret
KbdInputTrtE0S13:
cjne    A,#74h,KbdInputTrtE0S14
mov     KbdReadState,#KbdStateIdle  ; Touche Right
mov     A,#0
lcall   KbdPutch
mov     A,#4dh
lcall   KbdPutch
ret
KbdInputTrtE0S14:
cjne    A,#4ah,KbdInputTrtE0S15
mov     KbdReadState,#KbdStateIdle  ; Touche / pav num
mov     A,#2fh
lcall   KbdPutch
ret
KbdInputTrtE0S15:
cjne    A,#5ah,KbdInputTrtE0S16
mov     KbdReadState,#KbdStateIdle  ; Touche Enter (PavNum)
mov     A,#0dh
lcall   KbdPutch
ret

; Modifs faites suite pb sur pave intermediaire
KbdInputTrtE0S16:
cjne    A,#1fh,KbdInputTrtE0S17
mov     KbdReadState,#KbdStateIdle  ; Touche left window
ret
KbdInputTrtE0S17:
cjne    A,#27h,KbdInputTrtE0S18
mov     KbdReadState,#KbdStateIdle  ; Touche right window
ret
KbdInputTrtE0S18:
cjne    A,#2fh,KbdInputTrtE0S19
mov     KbdReadState,#KbdStateIdle  ; Touche taches window
ret
KbdInputTrtE0S19:
cjne    A,#7ch,KbdInputTrtE0S20
mov     KbdReadState,#KbdStateIdle  ; Touche print screen
ret
; Fin des modifs faites suite pb sur pave intermediaire

KbdInputTrtE0S20:
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

KbdInputTrtF0:                      ; Machine en etat F0
pop     ACC                         ; ------------------
cjne    A,#14h,KbdInputTrtF0S1
mov     KbdReadState,#KbdStateIdle  ; Fin Touche LCtl
mov     A,KbdReadModif
clr     KbdModifLCtl
mov     KbdReadModif,A
ret
KbdInputTrtF0S1:
cjne    A,#11h,KbdInputTrtF0S2
mov     KbdReadState,#KbdStateIdle  ; Fin Touche Alt
mov     A,KbdReadModif
clr     KbdModifAlt
mov     KbdReadModif,A
ret
KbdInputTrtF0S2:
cjne    A,#12h,KbdInputTrtF0S3
mov     KbdReadState,#KbdStateIdle  ; Fin Touche LShift
mov     A,KbdReadModif
clr     KbdModifLShift
mov     KbdReadModif,A
ret
KbdInputTrtF0S3:
cjne    A,#59h,KbdInputTrtF0S4
mov     KbdReadState,#KbdStateIdle  ; Fin Touche RShift
mov     A,KbdReadModif
clr     KbdModifRShift
mov     KbdReadModif,A
ret
KbdInputTrtF0S4:
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

KbdInputTrtE0F0:                    ; Machine en etat E0F0
pop     ACC                         ; --------------------
cjne    A,#11h,KbdInputTrtE0F0S1
mov     KbdReadState,#KbdStateIdle  ; Fin Touche AltGr
mov     A,KbdReadModif
clr     KbdModifAltgr
mov     KbdReadModif,A
ret
KbdInputTrtE0F0S1:
cjne    A,#14h,KbdInputTrtE0F0S2
mov     KbdReadState,#KbdStateIdle  ; Fin Touche RCtl
mov     A,KbdReadModif
clr     KbdModifRCtl
mov     KbdReadModif,A
ret
KbdInputTrtE0F0S2:
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

; Modifs faites suite pb sur pave intermediaire
;KbdInputTrtE012:                    ; Machine en etat E012
;pop     ACC                         ; --------------------
;cjne    A,#E0h,KbdInputTrtE012S1
;mov     KbdReadState,#KbdStateE012E0; Passage a l'etat E012E0
;ret
;KbdInputTrtE012S1:
;mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
;ret

;KbdInputTrtE012E0:                  ; Machine en etat E012E0
;pop     ACC                         ; ----------------------
;cjne    A,#1fh,KbdInputTrtE012E0S1
;mov     KbdReadState,#KbdStateIdle  ; Touche left window
;ret
;KbdInputTrtE012E0S1:
;cjne    A,#27h,KbdInputTrtE012E0S2
;mov     KbdReadState,#KbdStateIdle  ; Touche right window
;ret
;KbdInputTrtE012E0S2:
;cjne    A,#2fh,KbdInputTrtE012E0S3
;mov     KbdReadState,#KbdStateIdle  ; Touche taches window
;ret
;KbdInputTrtE012E0S3:
;cjne    A,#7ch,KbdInputTrtE012E0S4
;mov     KbdReadState,#KbdStateIdle  ; Touche print screen
;ret
;KbdInputTrtE012E0S4:
;mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
;ret
; Fin des modifs faites suite pb sur pave intermediaire

KbdInputTrtE1:                      ; Machine en etat E1
pop     ACC                         ; ------------------
cjne    A,#14h,KbdInputTrtE1S1
mov     KbdReadState,#KbdStateE114  ; Passage a l'etat E114
ret
KbdInputTrtE1S1:
cjne    A,#F0h,KbdInputTrtE1S2
mov     KbdReadState,#KbdStateE114  ; Passage a l'etat E1F0
ret
KbdInputTrtE1S2:
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

KbdInputTrtE114:                    ; Machine en etat E114
pop     ACC                         ; --------------------
cjne    A,#77h,KbdInputTrtE114S1
mov     KbdReadState,#KbdStateIdle  ; Traitement touche Pause Attn
ret
KbdInputTrtE114S1:
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

KbdInputTrtE1F0:                    ; Machine en etat E1F0
pop     ACC                         ; --------------------
mov     KbdReadState,#KbdStateIdle  ; Traitement par defaut
ret

KbdPutch:
;
; Titre     :   Ecriture d'un caractere dans le buffer de reception
;               ---------------------------------------------------
;
; Entree(s) :   Caractere dans le registre A
; Sortie(s) :   Rien
;               Met a jour l'index d'ecriture
;               Ne verifie pas si le buffer deborde
; Utilise   :   Registres DPH et DPL non sauvegardes
; Pile      :   3 avec l'appel
; Cycles    :   ?
;
push    ACC
mov     DPTR,#KbdBufferStart
mov     A,KbdIndexWrite
add     A,DPL
mov     DPL,A
mov     A,#0
addc    A,DPH
mov     DPH,A
pop     ACC
movx    @DPTR,A
inc     KbdIndexWrite
ret

KbdOutput:
;
; Titre     :   Sortie d'un octet vers le clavier
;               ---------------------------------
;
; Entree(s) :   Donnee a transmettre dans le registre A
; Sortie(s) :   Rien
;               A faire : time out ?
; Utilise   :   Registre B non sauvegarde, registre A restaure
; Pile      :   2 avec l'appel
; Cycles    :   ?
;
jnb     KbdClk,$        ; Attente clock a 1
jnb     KbdDat,$        ; Attente data a 1
clr     KbdEX           ; Arret interruptions
clr     KbdIT
clr     KbdClk
mov     B,#60
KbdOutputBoucle1:
djnz    B,KbdOutputBoucle1 ; Attente de 120 us
clr     KbdDat
setb    KbdClk
jb      KbdClk,$        ; Attente clock a 0
jnb     KbdClk,$        ; Attente clock a 1
mov     B,#8
KbdOutputBoucle2:
rrc     A
mov     KbdDat,C        ; Sortie des 8 bits LSB first
jb      KbdClk,$        ; Attente clock a 0
jnb     KbdClk,$        ; Attente clock a 1
djnz    B,KbdOutputBoucle2
rrc     A
mov     C,P
cpl     C
mov     KbdDat,C        ; Sortie du bit de parite
jb      KbdClk,$        ; Attente clock a 0
jnb     KbdClk,$        ; Attente clock a 1
setb    KbdDat
jb      KbdClk,$        ; Attente clock a 0
jnb     KbdClk,$        ; Attente clock a 1
setb    KbdIT
setb    KbdEX           ; Remise en route des IT
mov     B,#200
KbdOutputBoucle3:
nop
nop
nop
nop
nop
nop
nop
nop
djnz    B,KbdOutputBoucle3 ; Attente de 2 ms la reponse eventuelle du clavier
ret

KbdInputMap:
;
; Titre     :   Table de conversion entree clavier -> code
;               ------------------------------------------
; Organisation :
;   Code de touche par code de touche
;   Code normal, shift, ctl, alt
;   Alt l'emporte sur shift et ctl, ctl l'emporte sur shift
;   Chaque code en 2 bytes,
;       le premier etant a 0 pour un code etendu
;       le second n'ayant pas de signification si le premier est non nul
;       si egal a ffh, pas de code a sortir (premier ou deuxieme byte)
;
DW      ff00h,ff00h,ff00h,ff00h ; Touche 00h soit rien
DW      0043h,005ch,0066h,0070h ; Touche 01h soit F9
DW      ff00h,ff00h,ff00h,ff00h ; Touche 02h soit rien
DW      003fh,0058h,0062h,006ch ; Touche 03h soit F5
DW      003dh,0056h,0060h,006ah ; Touche 04h soit F3
DW      003bh,0054h,005eh,0068h ; Touche 05h soit F1
DW      003ch,0055h,005fh,0069h ; Touche 06h soit F2
DW      0086h,0088h,008ah,008ch ; Touche 07h soit F12
DW      ff00h,ff00h,ff00h,ff00h ; Touche 08h soit rien
DW      0044h,005dh,0067h,0071h ; Touche 09h soit F10
DW      0042h,005bh,0065h,006fh ; Touche 0ah soit F8
DW      0040h,0059h,0063h,006dh ; Touche 0bh soit F6
DW      003eh,0057h,0061h,006bh ; Touche 0ch soit F4
DW      0900h,000fh,0094h,ff00h ; Touche 0dh soit tab
DW      fd00h,0029h,ff00h,0029h ; Touche 0eh soit petit 2
DW      ff00h,ff00h,ff00h,ff00h ; Touche 0fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 10h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 11h soit Alt
DW      ff00h,ff00h,ff00h,ff00h ; Touche 12h soit LShift
DW      ff00h,ff00h,ff00h,ff00h ; Touche 13h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 14h soit LCtl
DW      6100h,4100h,0100h,001eh ; Touche 15h soit a
DW      2600h,3100h,ff00h,0078h ; Touche 16h soit &/1
DW      ff00h,ff00h,ff00h,ff00h ; Touche 17h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 18h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 19h soit rien
DW      7700h,5700h,1700h,0011h ; Touche 1ah soit w
DW      7300h,5300h,1300h,1f00h ; Touche 1bh soit s
DW      7100h,5100h,1100h,00ffh ; Touche 1ch soit q
DW      7a00h,5a00h,1a00h,002ch ; Touche 1dh soit z
DW      8200h,3200h,ff00h,7e00h ; Touche 1eh soit /2/~
DW      ff00h,ff00h,ff00h,ff00h ; Touche 1fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 20h soit rien
DW      6300h,4300h,0300h,002eh ; Touche 21h soit c
DW      7800h,5800h,1800h,002dh ; Touche 22h soit x
DW      6400h,4400h,0400h,0020h ; Touche 23h soit d
DW      6500h,4500h,0500h,0012h ; Touche 24h soit e
DW      2700h,3400h,ff00h,7b00h ; Touche 25h soit '/4/{
DW      2200h,3300h,ff00h,2300h ; Touche 26h soit "/3/#
DW      ff00h,ff00h,ff00h,ff00h ; Touche 27h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 28h soit rien
DW      2000h,2000h,2000h,2000h ; Touche 29h soit Space
DW      7600h,5600h,1600h,002fh ; Touche 2ah soit v
DW      6600h,4600h,0600h,0021h ; Touche 2bh soit f
DW      7400h,5400h,1400h,0014h ; Touche 2ch soit t
DW      7200h,5200h,1200h,ff00h ; Touche 2dh soit r
DW      2800h,3500h,ff00h,5b00h ; Touche 2eh soit (/5/[
DW      ff00h,ff00h,ff00h,ff00h ; Touche 2fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 30h soit rien
DW      6e00h,4e00h,0e00h,0031h ; Touche 31h soit n
DW      6200h,4200h,0200h,0030h ; Touche 32h soit b
DW      6800h,4800h,0800h,0023h ; Touche 33h soit h
DW      6700h,4700h,0700h,0022h ; Touche 34h soit g
DW      7900h,5900h,1900h,0015h ; Touche 35h soit y
DW      2d00h,3600h,1e00h,7c00h ; Touche 36h soit -/6/|
DW      ff00h,ff00h,ff00h,ff00h ; Touche 37h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 38h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 39h soit rien
DW      2c00h,3f00h,ff00h,ff00h ; Touche 3ah soit ,/?
DW      6a00h,4a00h,0a00h,0024h ; Touche 3bh soit j
DW      7500h,5500h,1500h,0016h ; Touche 3ch soit u
DW      8a00h,3700h,ff00h,6000h ; Touche 3dh soit /7/`
DW      5f00h,3800h,ff00h,5c00h ; Touche 3eh soit _/8/\
DW      ff00h,ff00h,ff00h,ff00h ; Touche 3fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 40h soit rien
DW      3b00h,2e00h,ff00h,0033h ; Touche 41h soit ;/.
DW      6b00h,4b00h,0b00h,0025h ; Touche 42h soit k
DW      6900h,4900h,0900h,0017h ; Touche 43h soit i
DW      6f00h,4f00h,0f00h,0018h ; Touche 44h soit o
DW      8500h,3000h,ff00h,4000h ; Touche 45h soit /0/@
DW      8700h,3900h,ff00h,5e00h ; Touche 46h soit /9/^
DW      ff00h,ff00h,ff00h,ff00h ; Touche 47h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 48h soit rien
DW      3a00h,2f00h,ff00h,0034h ; Touche 49h soit ://
DW      2100h,f500h,ff00h,0035h ; Touche 4ah soit !/
DW      6c00h,4c00h,0c00h,0026h ; Touche 4bh soit l
DW      6d00h,4d00h,0d00h,0032h ; Touche 4ch soit m
DW      7000h,5000h,1000h,0019h ; Touche 4dh soit p
DW      2900h,f800h,1f00h,5d00h ; Touche 4eh soit )//]
DW      ff00h,ff00h,ff00h,ff00h ; Touche 4fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 50h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 51h soit rien
DW      9700h,2500h,ff00h,0028h ; Touche 52h soit /%
DW      ff00h,ff00h,ff00h,ff00h ; Touche 53h soit rien
DW      ff00h,ff00h,1b00h,001ah ; Touche 54h soit circonflexe/trema
DW      3d00h,2b00h,ff00h,7d00h ; Touche 55h soit =/+/}
DW      ff00h,ff00h,ff00h,ff00h ; Touche 56h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 57h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 58h soit caps lock
DW      ff00h,ff00h,ff00h,ff00h ; Touche 59h soit RShift
DW      0d00h,0d00h,0a00h,001ch ; Touche 5ah soit entree (non num)
DW      2400h,9c00h,1d00h,cf00h ; Touche 5bh soit $//
DW      ff00h,ff00h,ff00h,ff00h ; Touche 5ch soit rien
DW      2a00h,e600h,1c00h,002bh ; Touche 5dh soit */
DW      ff00h,ff00h,ff00h,ff00h ; Touche 5eh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 5fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 60h soit rien
DW      3c00h,3e00h,ff00h,ff00h ; Touche 61h soit </>
DW      ff00h,ff00h,ff00h,ff00h ; Touche 62h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 63h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 64h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 65h soit rien
DW      0800h,0800h,7f00h,000eh ; Touche 66h soit back space
DW      ff00h,ff00h,ff00h,ff00h ; Touche 67h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 68h soit rien
DW      004fh,3100h,0075h,ff00h ; Touche 69h soit Num 1
DW      ff00h,ff00h,ff00h,ff00h ; Touche 6ah soit rien
DW      004bh,3400h,0073h,ff00h ; Touche 6bh soit Num 4
DW      0047h,3700h,0077h,ff00h ; Touche 6ch soit Num 7
DW      ff00h,ff00h,ff00h,ff00h ; Touche 6dh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 6eh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 6fh soit rien
DW      0052h,3000h,0092h,ff00h ; Touche 70h soit Num 0
DW      0053h,2e00h,0093h,ff00h ; Touche 71h soit Num .
DW      0050h,3200h,0091h,ff00h ; Touche 72h soit Num 2
DW      004ch,3500h,008fh,ff00h ; Touche 73h soit Num 5
DW      004dh,3600h,0074h,ff00h ; Touche 74h soit Num 6
DW      0048h,3800h,008dh,ff00h ; Touche 75h soit Num 8
DW      1b00h,1b00h,ff00h,ff00h ; Touche 76h soit Escape
DW      ff00h,ff00h,ff00h,ff00h ; Touche 77h soit Num lock
DW      0085h,0087h,0089h,008bh ; Touche 78h soit F11
DW      2b00h,2b00h,0090h,004eh ; Touche 79h soit Num +
DW      0051h,3300h,0076h,ff00h ; Touche 7ah soit Num 3
DW      2d00h,2d00h,008eh,004ah ; Touche 7bh soit Num -
DW      2a00h,2a00h,0096h,0037h ; Touche 7ch soit Num *
DW      0049h,3900h,0084h,ff00h ; Touche 7dh soit Num 9
DW      ff00h,ff00h,ff00h,ff00h ; Touche 7eh soit Screen lock
DW      ff00h,ff00h,ff00h,ff00h ; Touche 7fh soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 80h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 81h soit rien
DW      ff00h,ff00h,ff00h,ff00h ; Touche 82h soit rien
DW      0041h,005ah,0064h,006eh ; Touche 83h soit F7


; Recapitulatif des codes de sortie
; ---------------------------------

; Code : 00h -> Alt q
; Code : 01h -> Ctl a
; Code : 02h -> Ctl b
; Code : 03h -> Ctl c
; Code : 04h -> Ctl d
; Code : 05h -> Ctl e
; Code : 06h -> Ctl f
; Code : 07h -> Ctl g
; Code : 08h -> Back space ou Ctl h
; Code : 09h -> Tab ou Ctl i
; Code : 0ah -> Ctl Enter ou Ctl j
; Code : 0bh -> Ctl k
; Code : 0ch -> Ctl l
; Code : 0dh -> Enter ou Ctl m
; Code : 0eh -> Ctl n   ; Si 00h devant -> Alt Back space
; Code : 0fh -> Ctl o   ; Si 00h devant -> Shift Tab
; Code : 10h -> Ctl p
; Code : 11h -> Ctl q   ; Si 00h devant -> Alt w
; Code : 12h -> Ctl r   ; Si 00h devant -> Alt e
; Code : 13h -> Ctl s
; Code : 14h -> Ctl t   ; Si 00h devant -> Alt t
; Code : 15h -> Ctl u   ; Si 00h devant -> Alt y
; Code : 16h -> Ctl v   ; Si 00h devant -> Alt u
; Code : 17h -> Ctl w   ; Si 00h devant -> Alt i
; Code : 18h -> Ctl x   ; Si 00h devant -> Alt o
; Code : 19h -> Ctl y   ; Si 00h devant -> Alt p
; Code : 1ah -> Ctl z   ; Si 00h devant -> Alt ^/
; Code : 1bh -> Escape ou Ctl ^/
; Code : 1ch -> Ctl */ ; Si 00h devant -> Alt Enter
; Code : 1dh -> Ctl $//
; Code : 1eh -> Ctl -/6/|; Si 00h devant -> Alt a
; Code : 1fh -> Alt s   ; Si 00h devant -> Alt )//]
; Code : 20h -> Space   ; Si 00h devant -> Alt d
; Code : 21h -> !       ; Si 00h devant -> Alt f
; Code : 22h -> "       ; Si 00h devant -> Alt g
; Code : 23h -> #       ; Si 00h devant -> Alt h
; Code : 24h -> $       ; Si 00h devant -> Alt j
; Code : 25h -> %       ; Si 00h devant -> Alt k
; Code : 26h -> &       ; Si 00h devant -> Alt l
; Code : 27h -> '
; Code : 28h -> (       ; Si 00h devant -> Alt /%
; Code : 29h -> )       ; Si 00h devant -> Shift ou Alt petit 2
; Code : 2ah -> *
; Code : 2bh -> +       ; Si 00h devant -> Alt */
; Code : 2ch -> ,       ; Si 00h devant -> Alt z
; Code : 2dh -> -       ; Si 00h devant -> Alt x
; Code : 2eh -> .       ; Si 00h devant -> Alt c
; Code : 2fh -> /       ; Si 00h devant -> Alt v
; Code : 30h -> 0       ; Si 00h devant -> Alt b
; Code : 31h -> 1       ; Si 00h devant -> Alt n
; Code : 32h -> 2       ; Si 00h devant -> Alt m
; Code : 33h -> 3       ; Si 00h devant -> Alt ;/.
; Code : 34h -> 4       ; Si 00h devant -> Alt ://
; Code : 35h -> 5       ; Si 00h devant -> Alt !/
; Code : 36h -> 6
; Code : 37h -> 7       ; Si 00h devant -> Alt Num *
; Code : 38h -> 8
; Code : 39h -> 9
; Code : 3ah -> :
; Code : 3bh -> ;       ; Si 00h devant -> F1
; Code : 3ch -> <       ; Si 00h devant -> F2
; Code : 3dh -> =       ; Si 00h devant -> F3
; Code : 3eh -> >       ; Si 00h devant -> F4
; Code : 3fh -> ?       ; Si 00h devant -> F5
; Code : 40h -> @       ; Si 00h devant -> F6
; Code : 41h -> A       ; Si 00h devant -> F7
; Code : 42h -> B       ; Si 00h devant -> F8
; Code : 43h -> C       ; Si 00h devant -> F9
; Code : 44h -> D       ; Si 00h devant -> F10
; Code : 45h -> E
; Code : 46h -> F
; Code : 47h -> G       ; Si 00h devant -> Home
; Code : 48h -> H       ; Si 00h devant -> Up
; Code : 49h -> I       ; Si 00h devant -> Page up
; Code : 4ah -> J       ; Si 00h devant -> Alt Num -
; Code : 4bh -> K       ; Si 00h devant -> Left
; Code : 4ch -> L       ; Si 00h devant -> Num 5
; Code : 4dh -> M       ; Si 00h devant -> Right
; Code : 4eh -> N       ; Si 00h devant -> Alt Num +
; Code : 4fh -> O       ; Si 00h devant -> End
; Code : 50h -> P       ; Si 00h devant -> Down
; Code : 51h -> Q       ; Si 00h devant -> Page down
; Code : 52h -> R       ; Si 00h devant -> Insert
; Code : 53h -> S       ; Si 00h devant -> Delete
; Code : 54h -> T       ; Si 00h devant -> Shift F1
; Code : 55h -> U       ; Si 00h devant -> Shift F2
; Code : 56h -> V       ; Si 00h devant -> Shift F3
; Code : 57h -> W       ; Si 00h devant -> Shift F4
; Code : 58h -> X       ; Si 00h devant -> Shift F5
; Code : 59h -> Y       ; Si 00h devant -> Shift F6
; Code : 5ah -> Z       ; Si 00h devant -> Shift F7
; Code : 5bh -> [       ; Si 00h devant -> Shift F8
; Code : 5ch -> \       ; Si 00h devant -> Shift F9
; Code : 5dh -> ]       ; Si 00h devant -> Shift F10
; Code : 5eh -> ^       ; Si 00h devant -> Ctl F1
; Code : 5fh -> _       ; Si 00h devant -> Ctl F2
; Code : 60h -> `       ; Si 00h devant -> Ctl F3
; Code : 61h -> a       ; Si 00h devant -> Ctl F4
; Code : 62h -> b       ; Si 00h devant -> Ctl F5
; Code : 63h -> c       ; Si 00h devant -> Ctl F6
; Code : 64h -> d       ; Si 00h devant -> Ctl F7
; Code : 65h -> e       ; Si 00h devant -> Ctl F8
; Code : 66h -> f       ; Si 00h devant -> Ctl F9
; Code : 67h -> g       ; Si 00h devant -> Ctl F10
; Code : 68h -> h       ; Si 00h devant -> Alt F1
; Code : 69h -> i       ; Si 00h devant -> Alt F2
; Code : 6ah -> j       ; Si 00h devant -> Alt F3
; Code : 6bh -> k       ; Si 00h devant -> Alt F4
; Code : 6ch -> l       ; Si 00h devant -> Alt F5
; Code : 6dh -> m       ; Si 00h devant -> Alt F6
; Code : 6eh -> n       ; Si 00h devant -> Alt F7
; Code : 6fh -> o       ; Si 00h devant -> Alt F8
; Code : 70h -> p       ; Si 00h devant -> Alt F9
; Code : 71h -> q       ; Si 00h devant -> Alt F10
; Code : 72h -> r
; Code : 73h -> s       ; Si 00h devant -> Ctl Num 4
; Code : 74h -> t       ; Si 00h devant -> Ctl Num 6
; Code : 75h -> u       ; Si 00h devant -> Ctl Num 1
; Code : 76h -> v       ; Si 00h devant -> Ctl Num 3
; Code : 77h -> w       ; Si 00h devant -> Ctl Num 7
; Code : 78h -> x       ; Si 00h devant -> Alt &/1
; Code : 79h -> y
; Code : 7ah -> z
; Code : 7bh -> {
; Code : 7ch -> |
; Code : 7dh -> }
; Code : 7eh -> ~
; Code : 7fh -> Ctl Back space
; Code : 80h ->
; Code : 81h ->
; Code : 82h -> 
; Code : 83h ->
; Code : 84h ->         ; Si 00h devant -> Ctl Num 9
; Code : 85h ->        ; Si 00h devant -> F11
; Code : 86h ->         ; Si 00h devant -> F12
; Code : 87h ->        ; Si 00h devant -> Shift F11
; Code : 88h ->         ; Si 00h devant -> Shift F12
; Code : 89h ->         ; Si 00h devant -> Ctl F11
; Code : 8ah ->        ; Si 00h devant -> Ctl F12
; Code : 8bh ->         ; Si 00h devant -> Alt F11
; Code : 8ch ->         ; Si 00h devant -> Alt F12
; Code : 8dh ->         ; Si 00h devant -> Ctl Num 8
; Code : 8eh ->         ; Si 00h devant -> Ctl Num -
; Code : 8fh ->         ; Si 00h devant -> Ctl Num 5
; Code : 90h ->         ; Si 00h devant -> Ctl Num +
; Code : 91h ->         ; Si 00h devant -> Ctl Num 2
; Code : 92h ->         ; Si 00h devant -> Ctl Num 0
; Code : 93h ->         ; Si 00h devant -> Ctl Num .
; Code : 94h ->         ; Si 00h devant -> Ctl Tab
; Code : 95h ->
; Code : 96h ->         ; Si 00h devant -> Ctl Num *
; Code : 97h -> 
; Code : 98h ->
; Code : 99h ->
; Code : 9ah ->
; Code : 9bh ->
; Code : 9ch -> 
; Code : 9dh ->
; Code : 9eh ->
; Code : 9fh ->
; Code : a0h ->
; Code : a1h ->
; Code : a2h ->
; Code : a3h ->
; Code : a4h ->
; Code : a5h ->
; Code : a6h ->
; Code : a7h ->
; Code : a8h ->
; Code : a9h ->
; Code : aah ->
; Code : abh ->
; Code : ach ->
; Code : adh ->
; Code : aeh ->
; Code : afh ->
; Code : b0h ->
; Code : b1h ->
; Code : b2h ->
; Code : b3h ->
; Code : b4h ->
; Code : b5h ->
; Code : b6h ->
; Code : b7h ->
; Code : b8h ->
; Code : b9h ->
; Code : bah ->
; Code : bbh ->
; Code : bch ->
; Code : bdh ->
; Code : beh ->
; Code : bfh ->
; Code : c0h ->
; Code : c1h ->
; Code : c2h ->
; Code : c3h ->
; Code : c4h ->
; Code : c5h ->
; Code : c6h ->
; Code : c7h ->
; Code : c8h ->
; Code : c9h ->
; Code : cah ->
; Code : cbh ->
; Code : cch ->
; Code : cdh ->
; Code : ceh ->
; Code : cfh -> 
; Code : d0h ->
; Code : d1h ->
; Code : d2h ->
; Code : d3h ->
; Code : d4h ->
; Code : d5h ->
; Code : d6h ->
; Code : d7h ->
; Code : d8h ->
; Code : d9h ->
; Code : dah ->
; Code : dbh ->
; Code : dch ->
; Code : ddh ->
; Code : deh ->
; Code : dfh ->
; Code : e0h ->
; Code : e1h ->
; Code : e2h ->
; Code : e3h ->
; Code : e4h ->
; Code : e5h ->
; Code : e6h -> 
; Code : e7h ->
; Code : e8h ->
; Code : e9h ->
; Code : eah ->
; Code : ebh ->
; Code : ech ->
; Code : edh ->
; Code : eeh ->
; Code : efh ->
; Code : f0h ->
; Code : f1h ->
; Code : f2h ->
; Code : f3h ->
; Code : f4h ->
; Code : f5h -> 
; Code : f6h ->
; Code : f7h ->
; Code : f8h -> 
; Code : f9h ->
; Code : fah ->
; Code : fbh ->
; Code : fch ->
; Code : fdh -> Petit 2
; Code : feh ->
; Code : ffh ->
