// main.c

//-----------------------------------------------------------------------
//
//	Programme de test du PIC 18F4550 N 7: Transfert de donnes vers un 
//							afficheur 2*16 lignes en C
//							avec programme compatible du bootloader
//	---------------------------------------------------------------------
//
//	(C) 2008 CHAMOUARD
//  (Libre de droits pour utilisation non 
//	commerciale et  vos propres risques)
//
//-----------------------------------------------------------------------

// Le circuit correspond au montage le plus simple pour le 18F4550 :
//	- quartz 20 MHz avec ses deux capacits de 10 pF
//	- _MCLR tire au +5V
//	- Alimentation et dcouplage de 100 nF et 47F
//	- 470 nF sur Vusb
// L'afficheur est connect au PIC4550 via l'interface standard (voir interface centronics)
// avec :
//	- ADD0  ADD5 connect au port A du PIC (RA0  RA5)
//	- DAT0  DAT7 connect au port D du PIC (RD0  RD7)
//	- /RD connect au RE0 du PIC
//	- /WR connect au RE1 du PIC
//
// Nouvelle version pour MPLAB.X, par rapport  la config par dfaut propose par 
//   MPLAB, il faut mettre 0x1000 dans le Codeoffset :
//   Option du projet, option du linker XC8, additional options
//
// Au boot, le PIC initialise l'afficheur, met le curseur en route et affiche
//	"Hello world!"

#include <p18f4550.h>

#define	LcdCmdReadAdd	0x02
#define	LcdCmdWriteAdd	0x00
#define	LcdDataReadAdd	0x03
#define	LcdDataWriteAdd 0x01
#define	ClaAdd			0x04

//-----------------------------------------------------------------------
//
//	Initialisation du port I/O
//
//	Positionne les diffrents bits du bus et mise au repos
//
//-----------------------------------------------------------------------
void port_init(void)
{
	LATA=0;
	ADCON1=0x0F;				// A/D en digital i/o
	CMCON=0x07;					// Comparateurs en digital i/o
	TRISA=0xC0;					// Bits RA0  RA5 en sortie
	LATEbits.LATE0=1;
	LATEbits.LATE1=1;
	TRISEbits.TRISE0=0;			// /RD=1 et activ
	TRISEbits.TRISE1=0;			// /WR=1 et activ
	LATD=0;
	TRISD=0xFF;					// Bits RD0  RD7 en entre
}

//-----------------------------------------------------------------------
//
//	Lecture sur le port I/O
//
//	ADD sur 6 bits en entre
//	DAT en sortie sur 8 bits 
//
//-----------------------------------------------------------------------
unsigned char port_read(unsigned char ADD)
{
	unsigned char DAT;
	LATA=(0x3F & ADD);
	#asm
	nop
	nop
	nop
    nop
	#endasm
	LATEbits.LATE0=0;
	#asm
	nop
	nop
	nop
    nop
	#endasm
	DAT=PORTD;
	LATEbits.LATE0=1;
	return(DAT);
}

//-----------------------------------------------------------------------
//
//	Ecriture sur le port I/O
//
//	ADD sur 6 bits en entre
//	DAT sur 8 bits en entre 
//
//-----------------------------------------------------------------------
void port_write(unsigned char ADD, unsigned char DAT)
{
	LATA=(0x3F & ADD);
	LATD=DAT;
	TRISD=0;
	#asm
	nop
	nop
	nop
    nop
	#endasm
	LATEbits.LATE1=0;
	#asm
	nop
	nop
	nop
    nop
	#endasm
	LATEbits.LATE1=1;
	#asm
	nop
	nop
	nop
    nop
	#endasm
	TRISD=0xFF;
}

//-----------------------------------------------------------------------
//
//	Ecriture d'une commande  l'afficheur LCD
//
//	CMD sur 8 bits en entre
//
//-----------------------------------------------------------------------
void lcd_write_cmd(unsigned char CMD)
{
	while(port_read(LcdCmdReadAdd)&0x80);	// Attente afficheur pret	
	port_write(LcdCmdWriteAdd, CMD);		// Ecriture de la commande
}

//-----------------------------------------------------------------------
//
//	Ecriture d'une donne  l'afficheur LCD
//
//	DAT sur 8 bits en entre
//
//-----------------------------------------------------------------------
void lcd_write_data(unsigned char DAT)
{
	while(port_read(LcdCmdReadAdd)&0x80);	// Attente afficheur pret	
	port_write(LcdDataWriteAdd, DAT);		// Ecriture de la donne
}

void main(void)
{
	port_init();

	//LATA=0x15;						// Test adresse = 15h
	//TRISD=0;
	//LATD=0xC3;						// Test donne = C3h
	//LATEbits.LATE1=0;				// Test signal /WR

	lcd_write_cmd(0x38);			// Init du lcd
	lcd_write_cmd(0x01);			// Raz du lcd
	lcd_write_cmd(0x0F);			// Curseur ON
	lcd_write_data('H');
	lcd_write_data('e');
	lcd_write_data('l');
	lcd_write_data('l');
	lcd_write_data('o');
	lcd_write_data(' ');
	lcd_write_data('w');
	lcd_write_data('o');
	lcd_write_data('r');
	lcd_write_data('l');
	lcd_write_data('d');
	lcd_write_data(' ');
	lcd_write_data('!');
	
	while(1);
}
