// inter_ce.c

//-----------------------------------------------------------------------
//
//	Programme d'interface avec la carte inter_ce
//	--------------------------------------------
//
//	(C) 1994-2001 CHAMOUARD
//
//  (Libre de droits pour utilisation non commerciale 
//	et  vos propres risques)
//
//-----------------------------------------------------------------------

// Cette bibliotheque permet l'interfacage entre un PC et des cartes interface
//   microcotroleur via la carte inter_ce (interface_centronics)

// Configuration de la carte (avec port l'adresse de base de l'interface) :
//	- CKA (clock registre donnees) sur bit 1 de port+2 (/AUTO)
//	- CKB (clock registre add+cmd) sur bit 0 de port+2 (/STROBE)
//	- /MB (selection nibble haut/bas) sur bit 3 de port+2 (SELECT)
//	- D0 a D7 (bus de donnee) sur bits 0 a 7 de port (PD0 a PD7)
//	- RD0 (LSB lu) sur bit 4 de port+1 (ONLINE)
//	- RD1          sur bit 5 de port+1 (PE)
//	- RD2          sur bit 6 de port+1 (/ACQ)
//	- RD3 (MSB lu) sur bit 7 de port+1 (BUSY) avec inversion

// L'application doit :
//		- inclure inter_ce.h,
//		- initialiser le port de travail (defaut LPT2 donc 0x378),
//		- initialiser l'interface (permet un test de l'interface),
//		- utiliser les commandes de lecture et/ou d'ecriture.

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>

#include "..\libelec\inter_ce.h"

// Definitions des constantes, variables globales

#define CKA	2
#define	CKB 1
#define _MB 8

unsigned short port=0x378;

// Definition des sous-programmes internes

void ice_regA(int);
void ice_regD(int);
int	 ice_regI(void);

// Corps des fonctions

int ice_setport(int LPT)
//
//	Selection du numero de port Centronics de la carte INTER_CE
//
//	LPT=1, 2 ou 3
//  port positionne a 0x3BC,0x378 ou 0x278
//		(LPT2 est le port generalement utilise)
//  sortie 0 si LPT a une valeur valide, 1 sinon
//
{
	int out=0;
	switch(LPT)
	{
		case 1:	port=0x3BC;
			return(out);
		case 2:	port=0x378;
			return(out);
		case 3:	port=0x278;
			return(out);
		default: port=0x378;
	}
	return(1);
}

int ice_init()
//
//	Initialisation et test de la carte INTER_CE
//
//	sortie 0 si le test est OK, 1 sinon
//
{
	int i,j,erreur;
	ice_regA(0xe0);				// /TS=/WR=/RD=1
	ice_regA(0xc0);				// /TS=0 /WR=/RD=1
	erreur=0;
	for(i=0;i<256;++i)
	{
		ice_regD(i);			// Donnee i en sortie
		j=ice_regI();			// Lecture de la sortie
		//printf("%d %d\n",i, j);
		if(i!=j)erreur=1;
	}
	ice_regA(0xe0);				// /TS=/WR=/RD=1
	return(erreur);
}

void ice_write(int add,int data)
//
//	Ecriture d'une donnee via la carte INTER_CE
//
//	Donnee data ecrite a l'adresse add
//
{
	add=add&0x1f;
	ice_regD(data);				// Donnee sur le bus de sortie
	ice_regA(add+0xe0);			// /TS=/WR=/RD=1
	ice_regA(add+0x60);			// /WR=0 /TS=/RD=1
	ice_regA(add+0xe0);			// /TS=/WR=/RD=1
}

int ice_read(int add)
//
//	Lecture d'une donnee via la carte INTER_CE
//
//	retourne la donnee lue a l'adresse add
//
{
	int data;
	add=add&0x1f;
	ice_regA(add+0xe0);			// /TS=/WR=/RD=1
	ice_regA(add+0xa0);			// /RD=0 /TS=/RD=1
	data=ice_regI();			// Lecture de la donnee
	ice_regA(add+0xe0);			// /TS=/WR=/RD=1
	return(data);
}

//	Sous-programmes internes

void ice_regD(int data)
//
//	Ecriture dans le registre de donnees de la carte INTER_CE
//
{
	_outp((unsigned short)(port), data);	// data sur la sortie Centronics
	_outp((unsigned short)(port+2), CKA);	// CKA=0
	_outp((unsigned short)(port+2), 0);		// CKA=1
}

void ice_regA(int data)
//
//	Ecriture dans le registre d'adresse/ctl de la carte INTER_CE
//
{
	_outp((unsigned short)(port), data);	// data sur la sortie Centronics
	_outp((unsigned short)(port+2), CKB);	// CKB=0
	_outp((unsigned short)(port+2), 0);		// CKB=1
}

int ice_regI()
//
//	Lecture dans le port de sortie de la carte INTER_CE
//
{
	int j;
	_outp((unsigned short)(port+2), 0);		// Mise a 0 de /MB
	j=((_inp((unsigned short)(port+1))^128)>>4)&15;
	_outp((unsigned short)(port+2), _MB);	// Mise a 1 de /MB
	j=j+((_inp((unsigned short)(port+1))^128)&240);
	return(j);
}
