// i2cus.c

//-----------------------------------------------------------------------
//
//	Sous programmes d'interface I2C
//	-------------------------------
//
//	(C) 1996-2006 CHAMOUARD
//
//  (Libre de droits pour utilisation non commerciale 
//	et  vos propres risques)
//
//-----------------------------------------------------------------------

// Ces sous_programmes permettent d'interfacer un PC avec des composants sur
//	un bus I2C

// L'application doit fournir les routines d'interfacage avec le bus I2C :
//	- des lectures/ecritures bits a bit (i2c_putbit et i2c_getbit),
//	- des lectures/ecritures octet par octet (i2c_putbyte et i2c_getbyte),
//	- les ecriture des debut/fin de trame (i2c_start et i2c_stop).
//  selon les definitions ci-dessous.
// Un exemple de telles routines est disponible dans la bibliotheque i2c
//	qui dote un PC d'un bus i2c via la carte inter_ce.

// Ne pas oublier d'initialiser la liaison i2c (via un i2c_init() par exemple).

// Ce programme est operationnel pour :
//	- les E2PROM I2C type 24xCxx de 01 a 16, test pour :
//		- ST24C08B avec A2=0 ou 1
//		- PCF8582C-2 avec (A2,A1,A0)=0  7
//		- 24LC64 avec (A2,A1,A0)=0  7
//  - DS1621 en mode declench
//	- PCF8574 ou PCF8574A
//  - PCF8583 (horloge avec IT sur bus I2C)

// Pour les E2PROM la logique d'adressage est la suivante :
//  - si taille <=16kbits
//		l'adresse est code sur 11 bits
//		sur les tailles infrieures  16kbits, les bits de chip select
//		  (si supports par le circuit) permettent l'extension  16kbits
//		c'est le cas de la PCF8583 par exemple
//  - si taille >16kbits
//		l'adresse est code sur 16 bits
//		sur certains circuits les bits de chip select permettent l'extension
//		   4 Mbits
//		pour utiliser les bits de chip select, ajouter [CS2,CS1,CS0]*2^16
//		   l'adresse relle du circuit

// Include et definition des fonctions d'interface avec le bus i2c

#include <stdlib.h>
#include "..\libelec\i2cus.h"

int		i2c_init(void);
void	i2c_start(void);
void	i2c_stop(void);
void	i2c_putbit(int);
int		i2c_getbit(void);
void	i2c_putbyte(int);
int		i2c_getbyte(void);

//
// Fonctions d'interface avec des E2PROM I2C
//

int	i2c_e2promtype=0;

#define i2c_e2prom_standard	1
#define i2c_e2prom_etendu   2

//
//	Selection du type d'E2PROM connecte sur le bus i2c
//
void 	i2cus_e2prom_settype(int i2c_e2promtype_in)
{
	switch(i2c_e2promtype_in)
	{
		case ST24C08:
			i2c_e2promtype=i2c_e2prom_standard; // 16kbit max pas de ctl add
			break;
		case PCF8582:
			i2c_e2promtype=i2c_e2prom_standard; // 16kbit max pas de ctl add
			break;
		case M24LC64:
			i2c_e2promtype=i2c_e2prom_etendu; // 4M (512k + chip select) max pas de ctl add
			break;
		default:
			i2c_e2promtype=0;
			break;
	}
}

//
// Lecture d'un octet dans une E2PROM connectee au bus I2C
//
// Retourne :
//	0 si octet lu sans probleme
//  1 si pas d'acquitement dans un des tests
//  2 si le type d'E2PROM n'a pas ete defini ou est inconnu
//
int i2cus_e2prom_read(int add, int *data)
{
	if(i2c_e2promtype==0)return(2);
	*data=0;
	if(i2c_e2promtype==i2c_e2prom_standard)
	{
		add=add&0x7ff;
		i2c_start();					// Start
		i2c_putbyte(((add>>7)&0xfe)+0xa0);	// Control byte en ecriture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(add&0xff);			// Word address
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_start();					// Re-Start
		i2c_putbyte(((add>>7)&0xfe)+0xa1);	// Control byte en lecture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		*data=i2c_getbyte();			// Data
		i2c_getbit();					// No acknowledge
		i2c_stop();						// Stop
		return(0);
	}
	else if(i2c_e2promtype==i2c_e2prom_etendu)
	{
		add=add&0x7ffff;
		i2c_start();					// Start
		i2c_putbyte(((add>>15)&0xfe)+0xa0);	// Control byte en ecriture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte((add&0xff00)>>8);	// Word address MSB
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(add&0xff);			// Word address LSB
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_start();					// Re-Start
		i2c_putbyte(((add>>15)&0xfe)+0xa1);	// Control byte en lecture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		*data=i2c_getbyte();			// Data
		i2c_getbit();					// No acknowledge
		i2c_stop();						// Stop
		return(0);
	}
	return(2);
}

//
// Ecriture d'un octet dans une E2PROM connectee au bus I2C
//
// Retourne :
//	0 si octet ecrit sans probleme
//  1 si pas d'acquitement dans un des tests
//  2 si le type d'E2PROM n'a pas ete defini ou est inconnu
//
int i2cus_e2prom_write(int add, int data)
{
	if(i2c_e2promtype==0)return(2);
	if(i2c_e2promtype==i2c_e2prom_standard)
	{
		add=add&0x7ff;
		data=data&0xff;
		i2c_start();					// Start
		i2c_putbyte(((add>>7)&0xfe)+0xa0);	// Control byte en ecriture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(add&0xff);			// Word address
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(data);				// Data
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_stop();						// Stop
		do
		{
			i2c_start();                // Start
			i2c_putbyte(((add>>7)&0xfe)+0xa0);// Control byte en ecriture
		}
		while(i2c_getbit()!=0);			// Test acknowledge
		i2c_stop();
		return(0);
	}
	else if(i2c_e2promtype==i2c_e2prom_etendu)
	{
		add=add&0x7ffff;
		data=data&0xff;
		i2c_start();					// Start
		i2c_putbyte(((add>>15)&0xfe)+0xa0);	// Control byte en ecriture
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte((add&0xff00)>>8);	// Word address MSB
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(add&0xff);			// Word address LSB
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_putbyte(data);				// Data
		if(i2c_getbit()!=0)				// Acknowledge
		{
			i2c_stop();
			return(1);
		}
		i2c_stop();						// Stop
		do
		{
			i2c_start();                // Start
			i2c_putbyte(((add>>15)&0xfe)+0xa0);// Control byte en ecriture
		}
		while(i2c_getbit()!=0);			// Test acknowledge
		i2c_stop();
		return(0);
	}
	return(2);
}

//
// Fonctions d'interface avec un DS1621
//

// Declarations des fonctions internes
int i2cus_ds1621_readstatus(int add, int *res);
int i2cus_ds1621_writestatus(int add, int status);

//
//	Lecture du status du DS1621
//
//	0 si octet lu sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_ds1621_readstatus(int add, int *res)
{
	*res=0;
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xac);                  // Read configuration command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0x91+((add&0x07)<<1));	// DS1621 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	*res=i2c_getbyte();					// Data
	i2c_getbit();						// No acknowledge
	i2c_stop();							// Stop
	return(0);
}

//
//	Ecriture du status du DS1621
//
//	0 si octet ecrit sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_ds1621_writestatus(int add, int status)
{
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xac);                  // Write configuration command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(status);				// DS1621 config
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	return(0);
}

//
//	Initialisation du DS1621
//
//	Positionne le DS1621 en mode pop=0 1shot=1
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_ds1621_init(int add)
{
	int res;
	if(i2cus_ds1621_readstatus(add, &res)==1)return(1);
	if((res&0x03)==0x01)return(0);
	return(i2cus_ds1621_writestatus(add, 0x01));
}

//
//	Trigger du DS1621
//
//	Declenche une sequence de mesure sur le DS1621
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_ds1621_trigger(int add)
{
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xee);                  // Start conversion command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	return(0);
}

//
//	Verifie si une mesure est prete sur le DS1621
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
//	res est  1 si la mesure est prete
//
int	i2cus_ds1621_mesOK(int add, int *res)
{
	int resb;
	if(i2cus_ds1621_readstatus(add, &resb)!=0)
	{
		*res=1;
		return(1);
	}
	if((resb&0x80)==0)*res=0;
	else              *res=1;
	return(0);
}

//
//	Lecture de la temperature sur le DS1621
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
//	temp est la temperature en 256ime de degrs
//	Cette version  une rsolution du demi degrs
//
int	i2cus_ds1621_readtemp(int add, int *temp)
{
	int res1, res2;
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xaa);                  // Read temperature command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0x91+((add&0x07)<<1));	// DS1621 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	res1=i2c_getbyte();					// Data octet haut
	i2c_putbit(0);
	res2=i2c_getbyte();					// Data octet bas
	i2c_putbit(1);
	i2c_stop();
	*temp=(res1<<8)+res2;
	return(0);
}

//
//	Lecture de la temperature sur le DS1621
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
//	temp est la temperature en 256ime de degrs
//	Cette version  une rsolution du demi degrs
//
int	i2cus_ds1621_readtemp2(int add, int *temp)
{
	int res1, res2, res3;
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xaa);                  // Read temperature command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0x91+((add&0x07)<<1));	// DS1621 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	res1=i2c_getbyte();					// Data octet haut
	i2c_putbit(1);
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xa9);                  // Read slope command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0x91+((add&0x07)<<1));	// DS1621 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	res2=i2c_getbyte();					// Data slope
	i2c_putbit(1);
	i2c_start();
	i2c_putbyte(0x90+((add&0x07)<<1));	// DS1621 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0xa8);                  // Read counter command
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0x91+((add&0x07)<<1));	// DS1621 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	res3=i2c_getbyte();					// Data counter
	i2c_putbit(1);
	i2c_stop();
	*temp=((res1-1)<<8)+192+(int)(256.*(double)(res2-res3)/(double)(res2));
	return(0);
}


//
// Fonctions d'interface avec un PCF8574 (version - ou A)
//

//
// Ecriture d'un octet dans un PCF8574 (version - ou A)
//
// Retourne :
//	0 si octet ecrit sans probleme
//  1 si pas d'acquitement dans un des tests
//
int		i2cus_pcf8574_write(int add, int data)
{
	i2c_start();					// Start
	if(!(add&0x08))
		i2c_putbyte(((add<<1)&0x0e)+0x40);	// Control byte en ecriture, version -
	else
		i2c_putbyte(((add<<1)&0x0e)+0x70);	// Control byte en ecriture, version A
	if(i2c_getbit()!=0)				// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(data);				// Data
	if(i2c_getbit()!=0)				// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();						// Stop
	return(0);
}

//
// Lecture d'un octet dans un PCF8574 (version - ou A)
//
// Retourne :
//	0 si octet lu sans probleme
//  1 si pas d'acquitement dans un des tests
//
int		i2cus_pcf8574_read(int add, int *data)
{
	*data=0;
	i2c_start();					// Start
	if(!(add&0x08))
		i2c_putbyte(((add<<1)&0x0e)+0x41);	// Control byte en ecriture, version -
	else
		i2c_putbyte(((add<<1)&0x0e)+0x71);	// Control byte en ecriture, version A
	if(i2c_getbit()!=0)				// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	*data=i2c_getbyte();			// Data
	i2c_getbit();					// No acknowledge
	i2c_stop();						// Stop
	return(0);
}

//
// Fonctions d'interface avec un PCF8583
//

//
//	Initialisation du PCF8583
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
int		i2cus_pcf8583_init(int add)
{
	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // Control/status
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // 0x00 dans le control/status
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	return(0);
}

//
//	Ecriture de l'heure sur le PCF8583
//
//	Le PCF8583 est programme pour le 01/01 anne bixextile  l'heure/jour spcifi
//	pour un mode de fonctionnement sur 24 heures 
//	Le jour est entre 0 et 6
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_pcf8583_writeclock(int add, int weekday, int hour, int minute, int second)
{
	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // Control/status
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x80);                  // 0x80 dans le control/status (stop pendant prog)
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // Centiemes de secondes = 0
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	second=max(min(second,59),0);
	i2c_putbyte((second/10)*16+second%10);// Secondes
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	minute=max(min(minute,59),0);
	i2c_putbyte((minute/10)*16+minute%10);// Minutes
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	hour=max(min(hour,23),0);
	i2c_putbyte((hour/10)*16+hour%10);// Heures
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x01);					// Annees/date
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	weekday=max(min(weekday,6),0);
	i2c_putbyte(weekday<<5+1);			// Jour/mois
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // Control/status
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x00);                  // 0x00 dans le control/status
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	return(0);
}

//
//	Lecture de l'heure sur le PCF8583
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//
int	i2cus_pcf8583_readclock(int add, int *weekday, int *hour, int *minute, int *second)
{
	int res;

	*weekday=0;
	*hour=0;
	*minute=0;
	*second=0;

	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(0x02);                  // Seconde
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0xa1+((add&0x01)<<1));	// PCF8583 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	res=i2c_getbyte();					// Lecture secondes
	i2c_putbit(0);
	*second=10*(res>>4)+(res&0x0f);
	res=i2c_getbyte();					// Lecture minutes
	i2c_putbit(0);
	*minute=10*(res>>4)+(res&0x0f);
	res=i2c_getbyte();					// Lecture heures
	i2c_putbit(0);
	*hour=10*((res&0x3f)>>4)+(res&0x0f);
	res=i2c_getbyte();					// Lecture anne/date
	i2c_putbit(0);
	res=i2c_getbyte();					// Lecture jour/mois
	i2c_putbit(1);
	*weekday=res>>5;
	i2c_stop();
	return(0);
}

//
//	Ecriture dans la ram du PCF8583
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//	2 si adresse interdite en ecriture
//	loc doit etre entre 0x10 et 0xff
//
int	i2cus_pcf8583_writeram(int add, int loc, int dat)
{
	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(loc);					// Case adresse
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(dat);					// Donne  stoker
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_stop();
	return(0);
}

//
//	Lecture dans la ram du PCF8583
//
//	0 si operation sans probleme
//  1 si pas d'acquitement dans un des tests
//	loc peut prendre un valeur entre 0 et 255
//
int	i2cus_pcf8583_readram(int add, int loc, int *dat)
{
	*dat=0;
	i2c_start();
	i2c_putbyte(0xa0+((add&0x01)<<1));	// PCF8583 write
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_putbyte(loc);					// Case adresse
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	i2c_start();
	i2c_putbyte(0xa1+((add&0x01)<<1));	// PCF8583 read
	if(i2c_getbit()!=0)					// Acknowledge
	{
		i2c_stop();
		return(1);
	}
	*dat=i2c_getbyte();					// Lecture secondes
	i2c_putbit(1);
	i2c_stop();
	return(0);
}
